#!/bin/bash

# ================================================================
# ThangLQ Stack - Main Installation Script
# Standalone installer that downloads all required components
# ================================================================

# Default configuration - can be overridden by downloaded config.conf
BRAND_NAME="ThangLQ Stack"
BRAND_URL="https://thanglq.dev"
BRAND_VERSION="1.0.0"
SCRIPTS_BASE_URL="http://scripts.thanglq.dev"
SCRIPTS_CACHE_DIR="/tmp/thanglq-scripts"
AUTO_UPDATE_SCRIPTS="true"

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
CYAN='\033[0;36m'
WHITE='\033[1;37m'
NC='\033[0m'

# Get script directory
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"

# Basic logging function (before common.sh is loaded)
basic_log() {
    local level=$1
    local message=$2
    case $level in
        "ERROR") echo -e "${RED}[ERROR]${NC} $message" ;;
        "SUCCESS") echo -e "${GREEN}[SUCCESS]${NC} $message" ;;
        "WARNING") echo -e "${YELLOW}[WARNING]${NC} $message" ;;
        "INFO") echo -e "${BLUE}[INFO]${NC} $message" ;;
    esac
}

# Check if running as root
check_root() {
    if [[ $EUID -ne 0 ]]; then
        basic_log "ERROR" "Script must be run as root"
        echo "Please run: sudo $0"
        exit 1
    fi
}

# Check internet connectivity
check_internet() {
    basic_log "INFO" "Checking internet connectivity..."
    if ping -c 1 8.8.8.8 >/dev/null 2>&1 || ping -c 1 1.1.1.1 >/dev/null 2>&1; then
        basic_log "SUCCESS" "Internet connectivity confirmed"
        return 0
    else
        basic_log "ERROR" "No internet connectivity. Remote scripts require internet access."
        exit 1
    fi
}

# Install curl if not available
ensure_curl() {
    if ! command -v curl >/dev/null 2>&1; then
        basic_log "INFO" "Installing curl..."
        if command -v dnf >/dev/null 2>&1; then
            dnf install -y curl
        elif command -v yum >/dev/null 2>&1; then
            yum install -y curl
        else
            basic_log "ERROR" "Cannot find package manager to install curl"
            exit 1
        fi
    fi
}

# Download file with retry
download_file() {
    local url=$1
    local output=$2
    local retries=3
    local count=0
    
    # Create directory if needed
    mkdir -p "$(dirname "$output")"
    
    while [[ $count -lt $retries ]]; do
        if curl -f -s -L --connect-timeout 10 --max-time 30 "$url" -o "$output"; then
            return 0
        else
            ((count++))
            basic_log "WARNING" "Download failed (attempt $count/$retries): $(basename "$url")"
            sleep 2
        fi
    done
    
    return 1
}

# Download and cache script (enhanced version for install.sh)
download_script() {
    local script_path=$1
    local local_path="$SCRIPTS_CACHE_DIR/$script_path"
    local remote_url="$SCRIPTS_BASE_URL/$script_path"
    
    # Create directory structure
    mkdir -p "$(dirname "$local_path")"
    
    # Check if script exists and is recent (if auto-update enabled)
    if [[ -f "$local_path" ]] && [[ "$AUTO_UPDATE_SCRIPTS" != "true" ]]; then
        echo "$local_path"
        return 0
    fi
    
    # Check if script is older than 1 hour (if auto-update enabled)
    if [[ -f "$local_path" ]] && [[ "$AUTO_UPDATE_SCRIPTS" == "true" ]]; then
        local file_age=$(($(date +%s) - $(stat -c %Y "$local_path" 2>/dev/null || echo 0)))
        if [[ $file_age -lt 3600 ]]; then
            echo "$local_path"
            return 0
        fi
    fi
    
    # Use appropriate logging function
    if command -v log_message >/dev/null 2>&1; then
        log_message "INFO" "Downloading script: $script_path"
    else
        basic_log "INFO" "Downloading script: $script_path"
    fi
    
    # Download script with timeout and retry
    local retry_count=0
    local max_retries=3
    
    while [[ $retry_count -lt $max_retries ]]; do
        if curl -f -s -L --connect-timeout 10 --max-time 30 "$remote_url" -o "$local_path"; then
            chmod +x "$local_path"
            if command -v log_message >/dev/null 2>&1; then
                log_message "SUCCESS" "Downloaded: $script_path"
            else
                basic_log "SUCCESS" "Downloaded: $script_path"
            fi
            echo "$local_path"
            return 0
        else
            ((retry_count++))
            if command -v log_message >/dev/null 2>&1; then
                log_message "WARNING" "Download failed (attempt $retry_count/$max_retries): $script_path"
            else
                basic_log "WARNING" "Download failed (attempt $retry_count/$max_retries): $script_path"
            fi
            sleep 2
        fi
    done
    
    # If download failed, log error and return failure
    if command -v log_message >/dev/null 2>&1; then
        log_message "ERROR" "Failed to download script after $max_retries attempts: $script_path"
    else
        basic_log "ERROR" "Failed to download script after $max_retries attempts: $script_path"
    fi
    return 1
}

# Load remote module (enhanced version for install.sh)
load_remote_module() {
    local module_path=$1
    local script_file
    
    # Try to download the script first
    script_file=$(download_script "$module_path")
    local download_result=$?
    
    # Check if download was successful AND file exists
    if [[ $download_result -eq 0 ]] && [[ -n "$script_file" ]] && [[ -f "$script_file" ]]; then
        # Verify file is not empty
        if [[ -s "$script_file" ]]; then
            # Try to source the script
            if source "$script_file" 2>/dev/null; then
                if command -v log_message >/dev/null 2>&1; then
                    log_message "SUCCESS" "Loaded remote module: $module_path"
                else
                    basic_log "SUCCESS" "Loaded remote module: $module_path"
                fi
                return 0
            else
                if command -v log_message >/dev/null 2>&1; then
                    log_message "ERROR" "Failed to source module (syntax error?): $module_path"
                else
                    basic_log "ERROR" "Failed to source module (syntax error?): $module_path"
                fi
                return 1
            fi
        else
            if command -v log_message >/dev/null 2>&1; then
                log_message "ERROR" "Downloaded file is empty: $module_path"
            else
                basic_log "ERROR" "Downloaded file is empty: $module_path"
            fi
        fi
    fi
    
    # Fallback: Try to use local file if download failed
    local local_script="$SCRIPT_DIR/$module_path"
    if [[ -f "$local_script" ]]; then
        if command -v log_message >/dev/null 2>&1; then
            log_message "WARNING" "Remote download failed, using local file: $module_path"
        else
            basic_log "WARNING" "Remote download failed, using local file: $module_path"
        fi
        
        if [[ -s "$local_script" ]]; then
            if source "$local_script" 2>/dev/null; then
                if command -v log_message >/dev/null 2>&1; then
                    log_message "SUCCESS" "Loaded local module: $module_path"
                else
                    basic_log "SUCCESS" "Loaded local module: $module_path"
                fi
                return 0
            else
                if command -v log_message >/dev/null 2>&1; then
                    log_message "ERROR" "Failed to source local module: $module_path"
                else
                    basic_log "ERROR" "Failed to source local module: $module_path"
                fi
                return 1
            fi
        else
            if command -v log_message >/dev/null 2>&1; then
                log_message "ERROR" "Local file is empty: $module_path"
            else
                basic_log "ERROR" "Local file is empty: $module_path"
            fi
            return 1
        fi
    else
        if command -v log_message >/dev/null 2>&1; then
            log_message "ERROR" "Failed to load module (remote and local): $module_path"
        else
            basic_log "ERROR" "Failed to load module (remote and local): $module_path"
        fi
        return 1
    fi
}

# Check script availability
check_script_availability() {
    local script_path=$1
    local remote_url="$SCRIPTS_BASE_URL/$script_path"
    
    if curl -f -s -I --connect-timeout 5 --max-time 10 "$remote_url" >/dev/null 2>&1; then
        return 0
    else
        return 1
    fi
}

# Clear scripts cache
clear_scripts_cache() {
    if command -v log_message >/dev/null 2>&1; then
        log_message "INFO" "Clearing scripts cache..."
    else
        basic_log "INFO" "Clearing scripts cache..."
    fi
    rm -rf "$SCRIPTS_CACHE_DIR"
    mkdir -p "$SCRIPTS_CACHE_DIR"
    if command -v log_message >/dev/null 2>&1; then
        log_message "SUCCESS" "Scripts cache cleared"
    else
        basic_log "SUCCESS" "Scripts cache cleared"
    fi
}

# List available remote scripts
list_available_scripts() {
    local index_url="$SCRIPTS_BASE_URL/index.txt"
    
    if command -v log_message >/dev/null 2>&1; then
        log_message "INFO" "Fetching available scripts list..."
    else
        basic_log "INFO" "Fetching available scripts list..."
    fi
    
    if curl -f -s --connect-timeout 10 --max-time 30 "$index_url" 2>/dev/null; then
        return 0
    else
        if command -v log_message >/dev/null 2>&1; then
            log_message "WARNING" "Could not fetch scripts index"
        else
            basic_log "WARNING" "Could not fetch scripts index"
        fi
        return 1
    fi
}

# Download and load configuration
download_and_load_config() {
    local config_file="$SCRIPT_DIR/config.conf"
    local remote_config_url="$SCRIPTS_BASE_URL/config.conf"
    
    # Check if config exists locally
    if [[ ! -f "$config_file" ]]; then
        basic_log "INFO" "Downloading configuration file..."
        if download_file "$remote_config_url" "$config_file"; then
            basic_log "SUCCESS" "Configuration downloaded"
        else
            basic_log "WARNING" "Failed to download config, using defaults"
            return 1
        fi
    fi
    
    # Load configuration
    if [[ -f "$config_file" ]]; then
        source "$config_file"
        basic_log "INFO" "Configuration loaded"
        return 0
    else
        basic_log "WARNING" "Using built-in default configuration"
        return 1
    fi
}

# Download and load common utilities
download_and_load_common() {
    local utils_file="$SCRIPT_DIR/utils/common.sh"
    local remote_utils_url="$SCRIPTS_BASE_URL/utils/common.sh"
    
    # Check if utils exists locally
    if [[ ! -f "$utils_file" ]]; then
        basic_log "INFO" "Downloading common utilities..."
        if download_file "$remote_utils_url" "$utils_file"; then
            chmod +x "$utils_file"
            basic_log "SUCCESS" "Common utilities downloaded"
        else
            basic_log "ERROR" "Failed to download common utilities"
            exit 1
        fi
    fi
    
    # Load common utilities
    if [[ -f "$utils_file" ]]; then
        source "$utils_file"
        basic_log "SUCCESS" "Common utilities loaded"
        return 0
    else
        basic_log "ERROR" "Cannot load common utilities"
        exit 1
    fi
}

# Show initial banner
show_initial_banner() {
    clear
    echo -e "${CYAN}"
    echo "========================================================"
    echo "            $BRAND_NAME"
    echo "                   Version $BRAND_VERSION"
    echo "                   $BRAND_URL"
    echo "========================================================"
    echo -e "${NC}"
    echo -e "${WHITE}Standalone Remote Script Installer${NC}"
    echo -e "${WHITE}Auto-downloads components from: ${CYAN}$SCRIPTS_BASE_URL${NC}"
    echo ""
}

# Initialize all components
initialize() {
    show_initial_banner
    
    # Basic checks
    check_root
    check_internet
    ensure_curl
    
    # Download and load required files
    download_and_load_config
    download_and_load_common
    
    # Now we can use full functionality from common.sh
    create_directories
    detect_system
    
    # Show updated banner with loaded config
    show_banner
    echo -e "${WHITE}All components loaded successfully!${NC}"
    echo ""
    
    # Check for updates to core components
    check_for_updates
}

# Check for script updates
check_for_updates() {
    log_message "INFO" "Checking for script updates..."
    
    # Download latest config if available
    local config_script
    config_script=$(download_script "config.conf")
    if [[ $? -eq 0 ]] && [[ -f "$config_script" ]]; then
        # Copy to working directory if it's different
        if [[ "$config_script" != "$SCRIPT_DIR/config.conf" ]]; then
            cp "$config_script" "$SCRIPT_DIR/config.conf"
        fi
        log_message "INFO" "Configuration is up to date"
    fi
    
    # Download latest common utilities if available
    local common_script
    common_script=$(download_script "utils/common.sh")
    if [[ $? -eq 0 ]] && [[ -f "$common_script" ]]; then
        # Copy to working directory if it's different
        if [[ "$common_script" != "$SCRIPT_DIR/utils/common.sh" ]]; then
            mkdir -p "$SCRIPT_DIR/utils"
            cp "$common_script" "$SCRIPT_DIR/utils/common.sh"
            chmod +x "$SCRIPT_DIR/utils/common.sh"
        fi
        log_message "INFO" "Common utilities are up to date"
    fi
}

# Show main banner (from common.sh)
show_main_banner() {
    clear
    show_banner
    echo -e "${WHITE}Remote Modular LEMP Stack Installer${NC}"
    echo -e "${WHITE}Scripts downloaded on-demand from: ${CYAN}$SCRIPTS_BASE_URL${NC}"
    echo ""
}

# Show main menu
show_main_menu() {
    show_main_banner
    
    echo -e "${CYAN}=== INSTALLATION MODULES ===${NC}"
    echo -e "${WHITE}1)${NC}  Install Nginx (Standard)"
    echo -e "${WHITE}2)${NC}  Install OpenResty (Nginx + Lua)"
    echo -e "${WHITE}3)${NC}  Install PHP-FPM"
    echo -e "${WHITE}4)${NC}  Install MariaDB"
    echo -e "${WHITE}5)${NC}  Install Memcached"
    echo -e "${WHITE}6)${NC}  Install Redis"
    echo ""
    
    echo -e "${CYAN}=== UPGRADE/MANAGEMENT ===${NC}"
    echo -e "${WHITE}7)${NC}  Upgrade/Downgrade Nginx"
    echo -e "${WHITE}8)${NC}  Upgrade/Downgrade OpenResty"
    echo -e "${WHITE}9)${NC}  Upgrade/Downgrade PHP-FPM"
    echo -e "${WHITE}10)${NC} PHP Extensions Manager"
    echo ""
    
    echo -e "${CYAN}=== VIRTUAL HOSTS ===${NC}"
    echo -e "${WHITE}11)${NC} Create Virtual Host"
    echo -e "${WHITE}12)${NC} Delete Virtual Host"
    echo -e "${WHITE}13)${NC} List Virtual Hosts"
    echo ""
    
    echo -e "${CYAN}=== SYSTEM MANAGEMENT ===${NC}"
    echo -e "${WHITE}14)${NC} System Information"
    echo -e "${WHITE}15)${NC} Service Status"
    echo -e "${WHITE}16)${NC} View Logs"
    echo -e "${WHITE}17)${NC} Update Scripts Cache"
    echo ""
    
    # Commercial modules (if enabled)
    if [[ "$PERFORMANCE_MODULE_ENABLED" == "true" ]] || [[ "$SECURITY_MODULE_ENABLED" == "true" ]]; then
        echo -e "${CYAN}=== COMMERCIAL MODULES ===${NC}"
        if [[ "$PERFORMANCE_MODULE_ENABLED" == "true" ]]; then
            echo -e "${WHITE}18)${NC} Performance Tuning (Pro)"
        fi
        if [[ "$SECURITY_MODULE_ENABLED" == "true" ]]; then
            echo -e "${WHITE}19)${NC} Security Hardening (Pro)"
        fi
        echo ""
    fi
    
    echo -e "${WHITE}99)${NC} Exit"
    echo "========================================================"
    echo ""
}

# Install complete LEMP stack
install_complete_stack() {
    log_message "INFO" "Starting complete LEMP stack installation..."
    
    local total_steps=7
    local current_step=0
    
    # Step 1: Check requirements
    ((current_step++))
    show_progress $current_step $total_steps "Checking system requirements"
    check_requirements
    
    # Step 2: Install development tools
    ((current_step++))
    show_progress $current_step $total_steps "Installing development tools"
    install_development_tools
    
    # Step 3: Install web server (OpenResty by default)
    ((current_step++))
    show_progress $current_step $total_steps "Installing OpenResty"
    if load_remote_module "modules/web/openresty.sh"; then
        install_openresty
    else
        log_message "ERROR" "Failed to load OpenResty module"
        return 1
    fi
    
    # Step 4: Install PHP-FPM
    ((current_step++))
    show_progress $current_step $total_steps "Installing PHP-FPM"
    if load_remote_module "modules/web/php-fpm.sh"; then
        install_php_fpm
    else
        log_message "ERROR" "Failed to load PHP-FPM module"
        return 1
    fi
    
    # Step 5: Install MariaDB
    ((current_step++))
    show_progress $current_step $total_steps "Installing MariaDB"
    if load_remote_module "modules/database/mariadb.sh"; then
        install_mariadb
    else
        log_message "ERROR" "Failed to load MariaDB module"
        return 1
    fi
    
    # Step 6: Install cache systems
    ((current_step++))
    show_progress $current_step $total_steps "Installing cache systems"
    if load_remote_module "modules/cache/memcached.sh"; then
        install_memcached
    fi
    if load_remote_module "modules/cache/redis.sh"; then
        install_redis
    fi
    
    # Step 7: Start services
    ((current_step++))
    show_progress $current_step $total_steps "Starting services"
    start_all_services
    
    echo ""
    log_message "SUCCESS" "Complete LEMP stack installation finished!"
    
    show_installation_summary
}

# Start all services
start_all_services() {
    local services=("openresty" "php-fpm" "mariadb" "memcached" "redis")
    
    for service in "${services[@]}"; do
        if systemctl enable "$service" 2>/dev/null && systemctl start "$service" 2>/dev/null; then
            log_message "SUCCESS" "$service started successfully"
        else
            log_message "WARNING" "Failed to start $service"
        fi
    done
}

# Show installation summary
show_installation_summary() {
    echo ""
    echo -e "${GREEN}╔════════════════════════════════════════════════════════╗${NC}"
    echo -e "${GREEN}║              Installation Completed!                  ║${NC}"
    echo -e "${GREEN}╠════════════════════════════════════════════════════════╣${NC}"
    echo -e "${GREEN}║ OpenResty:       /usr/local/openresty/                ║${NC}"
    echo -e "${GREEN}║ PHP-FPM:         /usr/local/php*/                     ║${NC}"
    echo -e "${GREEN}║ MariaDB:         systemctl status mariadb             ║${NC}"
    echo -e "${GREEN}║ Memcached:       systemctl status memcached           ║${NC}"
    echo -e "${GREEN}║ Redis:           systemctl status redis               ║${NC}"
    echo -e "${GREEN}║                                                        ║${NC}"
    echo -e "${GREEN}║ Configuration:   $CONFIG_DIR/                   ║${NC}"
    echo -e "${GREEN}║ Logs:            $LOG_DIR/                     ║${NC}"
    echo -e "${GREEN}║ Virtual Hosts:   $VHOSTS_DIR/                   ║${NC}"
    echo -e "${GREEN}║ Scripts Cache:   $SCRIPTS_CACHE_DIR/                ║${NC}"
    echo -e "${GREEN}╚════════════════════════════════════════════════════════╝${NC}"
    echo ""
    echo "Next steps:"
    echo "• Create your first virtual host using option 11"
    echo "• Install PHP extensions using option 10"
    echo "• View system status using option 15"
}

# Handle individual component installation
install_component() {
    local component=$1
    
    case $component in
        "nginx")
            if load_remote_module "modules/core/nginx.sh"; then
                install_nginx
            else
                log_message "ERROR" "Failed to load Nginx module"
            fi
            ;;
        "openresty")
            if load_remote_module "modules/web/openresty.sh"; then
                install_openresty
            else
                log_message "ERROR" "Failed to load OpenResty module"
            fi
            ;;
        "php-fpm")
            echo "Available PHP versions: $PHP_AVAILABLE_VERSIONS"
            read -p "Enter PHP version to install [$PHP_DEFAULT_VERSION]: " php_version
            php_version=${php_version:-$PHP_DEFAULT_VERSION}
            if load_remote_module "modules/web/php-fpm.sh"; then
                install_php_fpm "$php_version"
            else
                log_message "ERROR" "Failed to load PHP-FPM module"
            fi
            ;;
        "mariadb")
            if load_remote_module "modules/database/mariadb.sh"; then
                install_mariadb
            else
                log_message "ERROR" "Failed to load MariaDB module"
            fi
            ;;
        "memcached")
            if load_remote_module "modules/cache/memcached.sh"; then
                install_memcached
            else
                log_message "ERROR" "Failed to load Memcached module"
            fi
            ;;
        "redis")
            if load_remote_module "modules/cache/redis.sh"; then
                install_redis
            else
                log_message "ERROR" "Failed to load Redis module"
            fi
            ;;
    esac
}

# Handle component upgrades
upgrade_component() {
    local component=$1
    
    case $component in
        "nginx")
            read -p "Enter new Nginx version: " new_version
            if load_remote_module "modules/core/nginx.sh"; then
                upgrade_nginx "$new_version"
            else
                log_message "ERROR" "Failed to load Nginx module"
            fi
            ;;
        "openresty")
            read -p "Enter new OpenResty version: " new_version
            if load_remote_module "modules/web/openresty.sh"; then
                upgrade_openresty "$new_version"
            else
                log_message "ERROR" "Failed to load OpenResty module"
            fi
            ;;
        "php-fpm")
            read -p "Enter current PHP version: " current_version
            read -p "Enter new PHP version: " new_version
            if load_remote_module "modules/web/php-fpm.sh"; then
                upgrade_php_fpm "$current_version" "$new_version"
            else
                log_message "ERROR" "Failed to load PHP-FPM module"
            fi
            ;;
    esac
}

# Virtual host management
manage_vhosts() {
    local action=$1
    
    case $action in
        "create")
            read -p "Enter domain name: " domain
            read -p "Enable SSL? (y/n) [y]: " ssl_choice
            ssl_choice=${ssl_choice:-y}
            if [[ "$ssl_choice" =~ ^[Yy]$ ]]; then
                ssl_enabled="true"
            else
                ssl_enabled="false"
            fi
            
            echo "Available PHP versions:"
            if load_remote_module "modules/web/php-fpm.sh"; then
                list_php_versions
            fi
            read -p "Enter PHP version [8.3]: " php_version
            php_version=${php_version:-8.3}
            
            if load_remote_module "scripts/vhost-manager.sh"; then
                create_vhost "$domain" "$ssl_enabled" "$php_version"
            else
                log_message "ERROR" "Failed to load VHost manager"
            fi
            ;;
        "delete")
            if load_remote_module "scripts/vhost-manager.sh"; then
                list_vhosts
                read -p "Enter domain to delete: " domain
                delete_vhost "$domain"
            else
                log_message "ERROR" "Failed to load VHost manager"
            fi
            ;;
        "list")
            if load_remote_module "scripts/vhost-manager.sh"; then
                list_vhosts
            else
                log_message "ERROR" "Failed to load VHost manager"
            fi
            ;;
    esac
}

# Show system information
show_system_info() {
    echo -e "${CYAN}=== System Information ===${NC}"
    echo "OS: $OS ($VER)"
    echo "Kernel: $(uname -r)"
    echo "Architecture: $(uname -m)"
    echo "CPU: $(nproc) cores"
    echo "Memory: $(free -h | awk 'NR==2{printf "%s/%s (%.2f%%)", $3,$2,$3*100/$2 }')"
    echo "Disk: $(df -h / | awk 'NR==2{printf "%s/%s (%s)", $3,$2,$5}')"
    echo "Uptime: $(uptime -p)"
    echo ""
    
    echo -e "${CYAN}=== Installed Software ===${NC}"
    [[ -f /usr/local/bin/nginx ]] && echo "Nginx: $(nginx -v 2>&1 | cut -d'/' -f2)" || echo "Nginx: Not installed"
    [[ -f /usr/local/bin/openresty ]] && echo "OpenResty: $(openresty -V 2>&1 | head -1 | cut -d'/' -f2)" || echo "OpenResty: Not installed"
    [[ -f /usr/local/bin/php ]] && echo "PHP: $(php -v | head -1 | cut -d' ' -f2)" || echo "PHP: Not installed"
    [[ $(systemctl is-active mariadb 2>/dev/null) == "active" ]] && echo "MariaDB: Active" || echo "MariaDB: Not active"
    [[ $(systemctl is-active memcached 2>/dev/null) == "active" ]] && echo "Memcached: Active" || echo "Memcached: Not active"
    [[ $(systemctl is-active redis 2>/dev/null) == "active" ]] && echo "Redis: Active" || echo "Redis: Not active"
    
    echo ""
    echo -e "${CYAN}=== Scripts Cache ===${NC}"
    echo "Cache directory: $SCRIPTS_CACHE_DIR"
    echo "Cached scripts: $(find "$SCRIPTS_CACHE_DIR" -name "*.sh" 2>/dev/null | wc -l)"
    echo "Cache size: $(du -sh "$SCRIPTS_CACHE_DIR" 2>/dev/null | cut -f1 || echo '0B')"
}

# Show service status
show_service_status() {
    echo -e "${CYAN}=== Service Status ===${NC}"
    
    local services=("nginx" "openresty" "php-fpm" "mariadb" "memcached" "redis")
    
    for service in "${services[@]}"; do
        local status=$(systemctl is-active "$service" 2>/dev/null || echo "not-found")
        case $status in
            "active")
                echo -e "$service: ${GREEN}✓ Running${NC}"
                ;;
            "inactive")
                echo -e "$service: ${YELLOW}○ Stopped${NC}"
                ;;
            "failed")
                echo -e "$service: ${RED}✗ Failed${NC}"
                ;;
            *)
                echo -e "$service: ${BLUE}○ Not installed${NC}"
                ;;
        esac
    done
}

# View logs
view_logs() {
    echo -e "${CYAN}=== Available Log Files ===${NC}"
    echo "1) Installation log"
    echo "2) Nginx error log"
    echo "3) PHP-FPM error log"
    echo "4) MariaDB error log"
    echo "5) System messages"
    echo ""
    
    read -p "Select log to view [1]: " log_choice
    log_choice=${log_choice:-1}
    
    case $log_choice in
        1) tail -f "$LOG_DIR/install.log" 2>/dev/null || echo "Log file not found" ;;
        2) tail -f "$LOG_DIR/nginx/error.log" 2>/dev/null || echo "Nginx log not found" ;;
        3) tail -f "$LOG_DIR/php*/fpm-error.log" 2>/dev/null || echo "PHP-FPM log not found" ;;
        4) tail -f /var/log/mariadb/mariadb.log 2>/dev/null || echo "MariaDB log not found" ;;
        5) tail -f /var/log/messages 2>/dev/null || echo "System log not found" ;;
        *) echo "Invalid choice" ;;
    esac
}

# Update scripts cache
update_scripts_cache() {
    echo -e "${CYAN}=== Scripts Cache Management ===${NC}"
    echo "1) Update all cached scripts"
    echo "2) Clear cache and re-download"
    echo "3) View cache status"
    echo "4) List available remote scripts"
    echo ""
    
    read -p "Select action [1]: " cache_choice
    cache_choice=${cache_choice:-1}
    
    case $cache_choice in
        1)
            log_message "INFO" "Updating cached scripts..."
            # Force re-download by temporarily enabling auto-update
            local original_auto_update="$AUTO_UPDATE_SCRIPTS"
            AUTO_UPDATE_SCRIPTS="true"
            
            # Update core files
            download_file "$SCRIPTS_BASE_URL/config.conf" "$SCRIPT_DIR/config.conf"
            download_file "$SCRIPTS_BASE_URL/utils/common.sh" "$SCRIPT_DIR/utils/common.sh"
            
            # Update common utilities
            download_script "utils/common.sh" >/dev/null
            
            # Re-download any previously cached scripts
            if [[ -d "$SCRIPTS_CACHE_DIR" ]]; then
                find "$SCRIPTS_CACHE_DIR" -name "*.sh" -type f | while read -r script; do
                    local relative_path="${script#$SCRIPTS_CACHE_DIR/}"
                    download_script "$relative_path" >/dev/null
                done
            fi
            
            AUTO_UPDATE_SCRIPTS="$original_auto_update"
            log_message "SUCCESS" "Scripts cache updated"
            ;;
        2)
            clear_scripts_cache
            log_message "SUCCESS" "Cache cleared. Scripts will be re-downloaded when needed."
            ;;
        3)
            show_system_info | grep -A5 "Scripts Cache"
            ;;
        4)
            echo "Available remote scripts:"
            list_available_scripts
            ;;
        *)
            echo "Invalid choice"
            ;;
    esac
}

# PHP extensions manager
manage_php_extensions() {
    echo -e "${CYAN}=== PHP Extensions Manager ===${NC}"
    echo "1) Install extension"
    echo "2) Remove extension"
    echo "3) List installed extensions"
    echo "4) Install common bundle"
    echo ""
    
    read -p "Select action [1]: " ext_choice
    ext_choice=${ext_choice:-1}
    
    if load_remote_module "scripts/php-extension-manager.sh"; then
        case $ext_choice in
            1)
                read -p "Enter extension name: " extension
                read -p "Enter PHP version [8.3]: " php_version
                php_version=${php_version:-8.3}
                install_pecl_extension "$extension" "$php_version"
                ;;
            2)
                read -p "Enter extension name: " extension
                read -p "Enter PHP version [8.3]: " php_version
                php_version=${php_version:-8.3}
                remove_extension "$extension" "$php_version"
                ;;
            3)
                read -p "Enter PHP version [8.3]: " php_version
                php_version=${php_version:-8.3}
                list_extensions "$php_version"
                ;;
            4)
                read -p "Enter PHP version [8.3]: " php_version
                php_version=${php_version:-8.3}
                install_common_bundle "$php_version"
                ;;
        esac
    else
        log_message "ERROR" "Failed to load PHP extensions manager"
    fi
}

# Load commercial modules
load_commercial_modules() {
    if [[ "$PERFORMANCE_MODULE_ENABLED" == "true" ]]; then
        if check_script_availability "modules/performance/performance.sh"; then
            load_remote_module "modules/performance/performance.sh"
        else
            log_message "WARNING" "Performance module enabled but not available remotely"
        fi
    fi
    
    if [[ "$SECURITY_MODULE_ENABLED" == "true" ]]; then
        if check_script_availability "modules/security/security.sh"; then
            load_remote_module "modules/security/security.sh"
        else
            log_message "WARNING" "Security module enabled but not available remotely"
        fi
    fi
}

# Main execution loop
main() {
    # Initialize everything (downloads config.conf and common.sh if needed)
    initialize
    
    if [[ "${1:-}" == "install" ]]; then
        install_complete_stack
        exit 0
    fi
    
    # Load commercial modules if available
    load_commercial_modules
    
    while true; do
        show_main_menu
        read -p "Enter your choice [1-99]: " choice
        
        case $choice in
            1) install_component "nginx" ;;
            2) install_component "openresty" ;;
            3) install_component "php-fpm" ;;
            4) install_component "mariadb" ;;
            5) install_component "memcached" ;;
            6) install_component "redis" ;;
            7) upgrade_component "nginx" ;;
            8) upgrade_component "openresty" ;;
            9) upgrade_component "php-fpm" ;;
            10) manage_php_extensions ;;
            11) manage_vhosts "create" ;;
            12) manage_vhosts "delete" ;;
            13) manage_vhosts "list" ;;
            14) show_system_info ;;
            15) show_service_status ;;
            16) view_logs ;;
            17) update_scripts_cache ;;
            18)
                if [[ "$PERFORMANCE_MODULE_ENABLED" == "true" ]]; then
                    performance_tuning_menu
                else
                    echo "Performance module not enabled"
                fi
                ;;
            19)
                if [[ "$SECURITY_MODULE_ENABLED" == "true" ]]; then
                    security_hardening_menu
                else
                    echo "Security module not enabled"
                fi
                ;;
            99)
                echo "Thank you for using $BRAND_NAME!"
                echo "Downloaded components:"
                echo "• Configuration: $SCRIPT_DIR/config.conf"
                echo "• Common utilities: $SCRIPT_DIR/utils/common.sh"
                echo "• Scripts cache: $SCRIPTS_CACHE_DIR"
                exit 0
                ;;
            *)
                echo "Invalid option. Please choose a valid number."
                sleep 2
                ;;
        esac
        
        # Pause before returning to menu
        echo ""
        read -p "Press Enter to continue..."
    done
}

# Error handling
trap 'echo -e "\n${RED}Installation interrupted${NC}"; exit 1' INT TERM

# Run main function
main "$@" 