#!/bin/bash

# ================================================================
# ThangLQ Stack - Memcached Module
# Memcached installation and management from source
# ================================================================

# Source common utilities
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
source "$SCRIPT_DIR/../../utils/common.sh"

# Install Memcached from source
install_memcached() {
    local version=${1:-$MEMCACHED_VERSION}
    
    log_message "INFO" "Installing Memcached $version from source..."
    
    cd /tmp
    
    # Install libevent dependency first
    install_libevent
    
    # Download and compile Memcached
    local memcached_url="http://www.memcached.org/files/memcached-${version}.tar.gz"
    wget "$memcached_url" -O "memcached-${version}.tar.gz"
    
    if [[ ! -f "memcached-${version}.tar.gz" ]]; then
        log_message "ERROR" "Failed to download Memcached"
        return 1
    fi
    
    tar -xzf "memcached-${version}.tar.gz"
    cd "memcached-${version}"
    
    # Configure Memcached
    ./configure \
        --prefix=/usr/local \
        --with-libevent=/usr/local \
        --enable-sasl \
        --enable-sasl-pwdb
    
    if [[ $? -ne 0 ]]; then
        log_message "ERROR" "Memcached configure failed"
        return 1
    fi
    
    # Compile and install
    make -j$(nproc) && make install
    
    if [[ $? -ne 0 ]]; then
        log_message "ERROR" "Memcached compilation/installation failed"
        return 1
    fi
    
    # Create memcached user
    create_user "$MEMCACHED_USER"
    
    # Create Memcached configuration
    create_memcached_config
    
    # Create systemd service
    create_memcached_service
    
    log_message "SUCCESS" "Memcached $version installed successfully"
    return 0
}

# Install libevent dependency
install_libevent() {
    log_message "INFO" "Installing libevent dependency..."
    
    cd /tmp
    
    # Download libevent
    wget https://github.com/libevent/libevent/releases/download/release-2.1.12-stable/libevent-2.1.12-stable.tar.gz
    tar -xzf libevent-2.1.12-stable.tar.gz
    cd libevent-2.1.12-stable
    
    ./configure --prefix=/usr/local
    make -j$(nproc) && make install
    
    # Update library path
    echo "/usr/local/lib" > /etc/ld.so.conf.d/libevent.conf
    ldconfig
    
    log_message "SUCCESS" "libevent installed successfully"
}

# Create Memcached configuration
create_memcached_config() {
    log_message "INFO" "Creating Memcached configuration..."
    
    # Calculate memory allocation based on system resources
    get_system_info
    local memcached_memory
    
    if [[ -n "$MEMCACHED_MEMORY" ]] && [[ "$MEMCACHED_MEMORY" != "256" ]]; then
        memcached_memory="$MEMCACHED_MEMORY"
    else
        # Calculate 15% of total RAM
        memcached_memory=$((TOTAL_RAM * 15 / 100))
        [[ $memcached_memory -lt 64 ]] && memcached_memory=64
        [[ $memcached_memory -gt 2048 ]] && memcached_memory=2048
    fi
    
    # Calculate max connections
    local max_connections=${MEMCACHED_CONNECTIONS:-$((CPU_CORES * 1024))}
    
    # Create configuration file
    cat > "$CONFIG_DIR/memcached.conf" << EOF
# ThangLQ Stack Memcached Configuration
# Generated on $(date)

# Basic settings
-p $MEMCACHED_PORT
-U 0
-m $memcached_memory
-c $max_connections
-t $CPU_CORES

# User and group
-u $MEMCACHED_USER

# Network settings
-l 127.0.0.1

# Memory settings
-M
-r

# Logging
-v
EOF

    log_message "SUCCESS" "Memcached configuration created"
}

# Create Memcached systemd service
create_memcached_service() {
    cat > /etc/systemd/system/memcached.service << EOF
[Unit]
Description=Memcached
Before=httpd.service
After=network.target

[Service]
Type=notify
User=$MEMCACHED_USER
ExecStart=/usr/local/bin/memcached -f $CONFIG_DIR/memcached.conf
ExecReload=/bin/kill -USR1 \$MAINPID
PIDFile=/var/run/memcached.pid
PrivateTmp=true

[Install]
WantedBy=multi-user.target
EOF

    systemctl daemon-reload
    systemctl enable memcached
    
    log_message "SUCCESS" "Memcached systemd service created"
}

# Start Memcached service
start_memcached() {
    log_message "INFO" "Starting Memcached service..."
    
    if systemctl start memcached; then
        log_message "SUCCESS" "Memcached started successfully"
        return 0
    else
        log_message "ERROR" "Failed to start Memcached"
        return 1
    fi
}

# Stop Memcached service
stop_memcached() {
    log_message "INFO" "Stopping Memcached service..."
    
    if systemctl stop memcached; then
        log_message "SUCCESS" "Memcached stopped successfully"
        return 0
    else
        log_message "ERROR" "Failed to stop Memcached"
        return 1
    fi
}

# Get Memcached status
memcached_status() {
    echo "=== Memcached Status ==="
    echo "Version: $(memcached -h 2>&1 | head -1 | awk '{print $2}' || echo 'Not available')"
    echo "Status: $(systemctl is-active memcached 2>/dev/null || echo 'Not installed')"
    echo "Port: $MEMCACHED_PORT"
    echo "Memory: $(grep '^-m' $CONFIG_DIR/memcached.conf 2>/dev/null | awk '{print $2}' || echo 'Unknown')MB"
    echo "Connections: $(grep '^-c' $CONFIG_DIR/memcached.conf 2>/dev/null | awk '{print $2}' || echo 'Unknown')"
    echo "Threads: $(grep '^-t' $CONFIG_DIR/memcached.conf 2>/dev/null | awk '{print $2}' || echo 'Unknown')"
    
    # Show stats if running
    if systemctl is-active memcached >/dev/null 2>&1; then
        echo ""
        echo "Live Stats:"
        echo "stats" | nc 127.0.0.1 $MEMCACHED_PORT 2>/dev/null | grep -E "(curr_connections|total_connections|cmd_get|cmd_set|get_hits|get_misses|bytes|uptime)" || echo "Unable to connect to Memcached"
    fi
}

# Test Memcached connection
test_memcached() {
    log_message "INFO" "Testing Memcached connection..."
    
    if command -v nc >/dev/null 2>&1; then
        # Test basic connection
        if echo "version" | nc 127.0.0.1 $MEMCACHED_PORT >/dev/null 2>&1; then
            log_message "SUCCESS" "Memcached connection test passed"
            
            # Test set/get operations
            echo "set test_key 0 60 5" | nc 127.0.0.1 $MEMCACHED_PORT >/dev/null 2>&1
            echo "hello" | nc 127.0.0.1 $MEMCACHED_PORT >/dev/null 2>&1
            
            local result=$(echo "get test_key" | nc 127.0.0.1 $MEMCACHED_PORT 2>/dev/null | grep "hello")
            if [[ -n "$result" ]]; then
                log_message "SUCCESS" "Memcached set/get test passed"
            else
                log_message "WARNING" "Memcached set/get test failed"
            fi
            
            # Cleanup
            echo "delete test_key" | nc 127.0.0.1 $MEMCACHED_PORT >/dev/null 2>&1
            
            return 0
        else
            log_message "ERROR" "Cannot connect to Memcached"
            return 1
        fi
    else
        log_message "WARNING" "netcat (nc) not available for testing"
        return 1
    fi
}

# Upgrade Memcached
upgrade_memcached() {
    local new_version=$1
    
    if [[ -z "$new_version" ]]; then
        log_message "ERROR" "Please specify Memcached version to upgrade to"
        return 1
    fi
    
    log_message "INFO" "Upgrading Memcached to version $new_version..."
    
    # Stop current Memcached
    stop_memcached
    
    # Backup current installation
    if [[ -f "/usr/local/bin/memcached" ]]; then
        cp "/usr/local/bin/memcached" "/usr/local/bin/memcached.backup.$(date +%Y%m%d_%H%M%S)"
    fi
    
    # Install new version
    if install_memcached "$new_version"; then
        log_message "SUCCESS" "Memcached upgraded to version $new_version successfully"
        start_memcached
        return 0
    else
        log_message "ERROR" "Failed to upgrade Memcached"
        return 1
    fi
}

# Flush Memcached data
flush_memcached() {
    log_message "INFO" "Flushing all Memcached data..."
    
    if echo "flush_all" | nc 127.0.0.1 $MEMCACHED_PORT >/dev/null 2>&1; then
        log_message "SUCCESS" "Memcached data flushed successfully"
        return 0
    else
        log_message "ERROR" "Failed to flush Memcached data"
        return 1
    fi
}

# Show Memcached configuration
show_memcached_config() {
    echo "=== Memcached Configuration ==="
    echo "Config file: $CONFIG_DIR/memcached.conf"
    echo "Service file: /etc/systemd/system/memcached.service"
    echo "Log files: /var/log/memcached/"
    echo ""
    echo "Current configuration:"
    if [[ -f "$CONFIG_DIR/memcached.conf" ]]; then
        cat "$CONFIG_DIR/memcached.conf"
    else
        echo "Configuration file not found"
    fi
}

# Main execution
case "${1:-help}" in
    "install")
        install_memcached "$2"
        ;;
    "start")
        start_memcached
        ;;
    "stop")
        stop_memcached
        ;;
    "restart")
        stop_memcached && start_memcached
        ;;
    "status")
        memcached_status
        ;;
    "test")
        test_memcached
        ;;
    "upgrade")
        upgrade_memcached "$2"
        ;;
    "flush")
        flush_memcached
        ;;
    "config")
        show_memcached_config
        ;;
    *)
        echo "ThangLQ Stack - Memcached Module"
        echo "Usage: $0 <command> [options]"
        echo ""
        echo "Commands:"
        echo "  install [version]  - Install Memcached from source"
        echo "  start              - Start Memcached service"
        echo "  stop               - Stop Memcached service"
        echo "  restart            - Restart Memcached service"
        echo "  status             - Show Memcached status"
        echo "  test               - Test Memcached connection"
        echo "  upgrade <version>  - Upgrade Memcached"
        echo "  flush              - Flush all cached data"
        echo "  config             - Show configuration"
        echo ""
        echo "Examples:"
        echo "  $0 install 1.6.22"
        echo "  $0 status"
        echo "  $0 test"
        echo "  $0 flush"
        ;;
esac 