#!/bin/bash

# ================================================================
# ThangLQ Stack - Redis Module
# Redis installation and management from source
# ================================================================

# Source common utilities
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
source "$SCRIPT_DIR/../../utils/common.sh"

# Install Redis from source
install_redis() {
    local version=${1:-$REDIS_VERSION}
    
    log_message "INFO" "Installing Redis $version from source..."
    
    cd /tmp
    
    # Download and compile Redis
    local redis_url="https://download.redis.io/releases/redis-${version}.tar.gz"
    wget "$redis_url" -O "redis-${version}.tar.gz"
    
    if [[ ! -f "redis-${version}.tar.gz" ]]; then
        log_message "ERROR" "Failed to download Redis"
        return 1
    fi
    
    tar -xzf "redis-${version}.tar.gz"
    cd "redis-${version}"
    
    # Compile Redis
    make -j$(nproc)
    if [[ $? -ne 0 ]]; then
        log_message "ERROR" "Redis compilation failed"
        return 1
    fi
    
    # Install Redis
    make install PREFIX=/usr/local/redis
    if [[ $? -ne 0 ]]; then
        log_message "ERROR" "Redis installation failed"
        return 1
    fi
    
    # Create symlinks
    ln -sf /usr/local/redis/bin/redis-server /usr/local/bin/redis-server
    ln -sf /usr/local/redis/bin/redis-cli /usr/local/bin/redis-cli
    ln -sf /usr/local/redis/bin/redis-benchmark /usr/local/bin/redis-benchmark
    
    # Create redis user
    create_user "$REDIS_USER"
    
    # Create Redis configuration
    create_redis_config
    
    # Create systemd service
    create_redis_service
    
    log_message "SUCCESS" "Redis $version installed successfully"
    return 0
}

# Create Redis configuration
create_redis_config() {
    log_message "INFO" "Creating Redis configuration..."
    
    # Calculate maxmemory based on system resources
    get_system_info
    local max_memory
    
    if [[ -n "$REDIS_MEMORY" ]] && [[ "$REDIS_MEMORY" != "256" ]]; then
        max_memory="${REDIS_MEMORY}mb"
    else
        # Calculate 10% of total RAM
        local redis_memory=$((TOTAL_RAM * 10 / 100))
        [[ $redis_memory -lt 64 ]] && redis_memory=64
        [[ $redis_memory -gt 1024 ]] && redis_memory=1024
        max_memory="${redis_memory}mb"
    fi
    
    # Create directories
    mkdir -p "$CONFIG_DIR" /var/lib/redis /var/log/redis
    chown "$REDIS_USER:$REDIS_USER" /var/lib/redis /var/log/redis
    
    # Create configuration file
    cat > "$CONFIG_DIR/redis.conf" << EOF
# ThangLQ Stack Redis Configuration
# Generated on $(date)

# Network settings
bind $REDIS_BIND_ADDRESS
port $REDIS_PORT
tcp-backlog 511
timeout 0
tcp-keepalive 300

# General settings
daemonize yes
supervised systemd
pidfile /var/run/redis_${REDIS_PORT}.pid
loglevel notice
logfile /var/log/redis/redis-server.log
databases $REDIS_DATABASES

# Memory settings
maxmemory $max_memory
maxmemory-policy allkeys-lru
maxmemory-samples 5

# Persistence settings
save 900 1
save 300 10
save 60 10000
stop-writes-on-bgsave-error yes
rdbcompression yes
rdbchecksum yes
dbfilename dump.rdb
dir /var/lib/redis

# Replication settings
replica-serve-stale-data yes
replica-read-only yes
repl-diskless-sync no
repl-diskless-sync-delay 5
repl-ping-replica-period 10
repl-timeout 60
repl-disable-tcp-nodelay no
repl-backlog-size 1mb
repl-backlog-ttl 3600

# Security settings
rename-command FLUSHDB ""
rename-command FLUSHALL ""
rename-command EVAL ""
rename-command DEBUG ""
rename-command CONFIG ""
EOF

    # Add password if configured
    if [[ -n "$REDIS_PASSWORD" ]]; then
        echo "requirepass $REDIS_PASSWORD" >> "$CONFIG_DIR/redis.conf"
    else
        # Generate random password
        REDIS_PASSWORD=$(generate_password 16)
        echo "requirepass $REDIS_PASSWORD" >> "$CONFIG_DIR/redis.conf"
        log_message "INFO" "Generated Redis password: $REDIS_PASSWORD"
        echo "Redis password: $REDIS_PASSWORD" >> "$LOG_DIR/credentials.txt"
    fi
    
    # Add performance optimizations
    cat >> "$CONFIG_DIR/redis.conf" << EOF

# Performance optimizations
hz 10
dynamic-hz yes
aof-rewrite-incremental-fsync yes
rdb-save-incremental-fsync yes

# Client settings
client-output-buffer-limit normal 0 0 0
client-output-buffer-limit replica 256mb 64mb 60
client-output-buffer-limit pubsub 32mb 8mb 60
client-query-buffer-limit 1gb

# Slow log settings
slowlog-log-slower-than 10000
slowlog-max-len 128

# Latency monitoring
latency-monitor-threshold 100
EOF

    log_message "SUCCESS" "Redis configuration created"
}

# Create Redis systemd service
create_redis_service() {
    cat > /etc/systemd/system/redis.service << EOF
[Unit]
Description=Advanced key-value store
After=network.target
Documentation=http://redis.io/documentation, man:redis-server(1)

[Service]
Type=notify
ExecStart=/usr/local/bin/redis-server $CONFIG_DIR/redis.conf
ExecReload=/bin/kill -USR2 \$MAINPID
ExecStop=/usr/local/bin/redis-cli shutdown
TimeoutStopSec=0
Restart=always
User=$REDIS_USER
Group=$REDIS_USER
RuntimeDirectory=redis
RuntimeDirectoryMode=0755

# Security settings
NoNewPrivileges=true
PrivateTmp=true
PrivateDevices=true
ProtectHome=true
ProtectSystem=strict
ReadWritePaths=/var/lib/redis
ReadWritePaths=/var/log/redis

[Install]
WantedBy=multi-user.target
EOF

    systemctl daemon-reload
    systemctl enable redis
    
    log_message "SUCCESS" "Redis systemd service created"
}

# Start Redis service
start_redis() {
    log_message "INFO" "Starting Redis service..."
    
    if systemctl start redis; then
        log_message "SUCCESS" "Redis started successfully"
        return 0
    else
        log_message "ERROR" "Failed to start Redis"
        return 1
    fi
}

# Stop Redis service
stop_redis() {
    log_message "INFO" "Stopping Redis service..."
    
    if systemctl stop redis; then
        log_message "SUCCESS" "Redis stopped successfully"
        return 0
    else
        log_message "ERROR" "Failed to stop Redis"
        return 1
    fi
}

# Get Redis status
redis_status() {
    echo "=== Redis Status ==="
    echo "Version: $(redis-server --version 2>/dev/null | awk '{print $3}' | cut -d'=' -f2 || echo 'Not available')"
    echo "Status: $(systemctl is-active redis 2>/dev/null || echo 'Not installed')"
    echo "Port: $REDIS_PORT"
    echo "Bind Address: $REDIS_BIND_ADDRESS"
    echo "Max Memory: $(grep '^maxmemory' $CONFIG_DIR/redis.conf 2>/dev/null | awk '{print $2}' || echo 'Unknown')"
    echo "Databases: $REDIS_DATABASES"
    
    # Show Redis info if running
    if systemctl is-active redis >/dev/null 2>&1; then
        echo ""
        echo "Live Stats:"
        if [[ -n "$REDIS_PASSWORD" ]]; then
            redis-cli -a "$REDIS_PASSWORD" --no-auth-warning INFO server 2>/dev/null | grep -E "(redis_version|uptime_in_seconds|connected_clients|used_memory_human)" || echo "Unable to connect to Redis"
        else
            redis-cli INFO server 2>/dev/null | grep -E "(redis_version|uptime_in_seconds|connected_clients|used_memory_human)" || echo "Unable to connect to Redis"
        fi
    fi
}

# Test Redis connection
test_redis() {
    log_message "INFO" "Testing Redis connection..."
    
    local redis_cmd="redis-cli"
    if [[ -n "$REDIS_PASSWORD" ]]; then
        redis_cmd="redis-cli -a $REDIS_PASSWORD --no-auth-warning"
    fi
    
    # Test basic connection
    if $redis_cmd ping >/dev/null 2>&1; then
        log_message "SUCCESS" "Redis connection test passed"
        
        # Test set/get operations
        if $redis_cmd set test_key "hello_from_thanglq_stack" >/dev/null 2>&1; then
            local result=$($redis_cmd get test_key 2>/dev/null)
            if [[ "$result" == "hello_from_thanglq_stack" ]]; then
                log_message "SUCCESS" "Redis set/get test passed"
            else
                log_message "WARNING" "Redis set/get test failed"
            fi
            
            # Cleanup
            $redis_cmd del test_key >/dev/null 2>&1
        else
            log_message "WARNING" "Redis set operation failed"
        fi
        
        return 0
    else
        log_message "ERROR" "Cannot connect to Redis"
        return 1
    fi
}

# Upgrade Redis
upgrade_redis() {
    local new_version=$1
    
    if [[ -z "$new_version" ]]; then
        log_message "ERROR" "Please specify Redis version to upgrade to"
        return 1
    fi
    
    log_message "INFO" "Upgrading Redis to version $new_version..."
    
    # Stop current Redis
    stop_redis
    
    # Backup current installation
    if [[ -d "/usr/local/redis" ]]; then
        mv "/usr/local/redis" "/usr/local/redis.backup.$(date +%Y%m%d_%H%M%S)"
    fi
    
    # Install new version
    if install_redis "$new_version"; then
        log_message "SUCCESS" "Redis upgraded to version $new_version successfully"
        start_redis
        return 0
    else
        log_message "ERROR" "Failed to upgrade Redis"
        return 1
    fi
}

# Flush Redis data
flush_redis() {
    log_message "INFO" "Flushing all Redis data..."
    
    read -p "Are you sure you want to flush all Redis data? (yes/no): " confirm
    if [[ "$confirm" == "yes" ]]; then
        local redis_cmd="redis-cli"
        if [[ -n "$REDIS_PASSWORD" ]]; then
            redis_cmd="redis-cli -a $REDIS_PASSWORD --no-auth-warning"
        fi
        
        if $redis_cmd flushall >/dev/null 2>&1; then
            log_message "SUCCESS" "Redis data flushed successfully"
            return 0
        else
            log_message "ERROR" "Failed to flush Redis data"
            return 1
        fi
    else
        log_message "INFO" "Operation cancelled"
        return 0
    fi
}

# Show Redis configuration
show_redis_config() {
    echo "=== Redis Configuration ==="
    echo "Config file: $CONFIG_DIR/redis.conf"
    echo "Service file: /etc/systemd/system/redis.service"
    echo "Data directory: /var/lib/redis"
    echo "Log directory: /var/log/redis"
    echo ""
    echo "Current configuration:"
    if [[ -f "$CONFIG_DIR/redis.conf" ]]; then
        cat "$CONFIG_DIR/redis.conf"
    else
        echo "Configuration file not found"
    fi
}

# Main execution
case "${1:-help}" in
    "install")
        install_redis "$2"
        ;;
    "start")
        start_redis
        ;;
    "stop")
        stop_redis
        ;;
    "restart")
        stop_redis && start_redis
        ;;
    "status")
        redis_status
        ;;
    "test")
        test_redis
        ;;
    "upgrade")
        upgrade_redis "$2"
        ;;
    "flush")
        flush_redis
        ;;
    "config")
        show_redis_config
        ;;
    *)
        echo "ThangLQ Stack - Redis Module"
        echo "Usage: $0 <command> [options]"
        echo ""
        echo "Commands:"
        echo "  install [version]  - Install Redis from source"
        echo "  start              - Start Redis service"
        echo "  stop               - Stop Redis service"
        echo "  restart            - Restart Redis service"
        echo "  status             - Show Redis status"
        echo "  test               - Test Redis connection"
        echo "  upgrade <version>  - Upgrade Redis"
        echo "  flush              - Flush all Redis data"
        echo "  config             - Show configuration"
        echo ""
        echo "Examples:"
        echo "  $0 install 7.2.4"
        echo "  $0 status"
        echo "  $0 test"
        echo "  $0 flush"
        ;;
esac 