#!/bin/bash

# ================================================================
# ThangLQ Stack - Nginx Module
# Standard Nginx installation and management
# ================================================================

# Source common utilities
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
source "$SCRIPT_DIR/../../utils/common.sh"

# Install standard Nginx
install_nginx() {
    local version=${1:-$NGINX_VERSION}
    
    log_message "INFO" "Installing Nginx $version from source..."
    
    cd /tmp
    
    # Download Nginx source
    local nginx_url="http://nginx.org/download/nginx-${version}.tar.gz"
    wget "$nginx_url" -O "nginx-${version}.tar.gz"
    
    if [[ ! -f "nginx-${version}.tar.gz" ]]; then
        log_message "ERROR" "Failed to download Nginx"
        return 1
    fi
    
    tar -xzf "nginx-${version}.tar.gz"
    cd "nginx-${version}"
    
    # Configure Nginx
    ./configure \
        --prefix=/usr/local/nginx \
        --user=$NGINX_USER \
        --group=$NGINX_GROUP \
        --with-http_ssl_module \
        --with-http_v2_module \
        --with-http_realip_module \
        --with-http_stub_status_module \
        --with-http_gzip_static_module \
        --with-pcre \
        --with-file-aio \
        --with-http_secure_link_module \
        --with-threads \
        $NGINX_CONFIGURE_ARGS
    
    if [[ $? -ne 0 ]]; then
        log_message "ERROR" "Nginx configure failed"
        return 1
    fi
    
    # Compile Nginx
    make -j$(nproc)
    if [[ $? -ne 0 ]]; then
        log_message "ERROR" "Nginx compilation failed"
        return 1
    fi
    
    # Install Nginx
    make install
    if [[ $? -ne 0 ]]; then
        log_message "ERROR" "Nginx installation failed"
        return 1
    fi
    
    # Create symlinks
    ln -sf /usr/local/nginx/sbin/nginx /usr/local/bin/nginx
    
    # Create nginx user
    create_user "$NGINX_USER"
    
    # Create Nginx configuration
    create_nginx_config
    
    # Create systemd service
    create_nginx_service
    
    log_message "SUCCESS" "Nginx $version installed successfully"
    return 0
}

# Create Nginx configuration
create_nginx_config() {
    local nginx_conf="/usr/local/nginx/conf/nginx.conf"
    local conf_dir="/usr/local/nginx/conf"
    
    # Create directories
    mkdir -p "$conf_dir"/{conf.d,ssl,includes}
    mkdir -p "$CACHE_DIR"/{fastcgi,proxy,scgi,uwsgi}
    
    # Backup existing configuration
    backup_file "$nginx_conf"
    
    # Create main configuration
    cat > "$nginx_conf" << EOF
user $NGINX_USER;
worker_processes $NGINX_WORKER_PROCESSES;
worker_rlimit_nofile 65535;
error_log $LOG_DIR/nginx/error.log $NGINX_ERROR_LOG_LEVEL;
pid /var/run/nginx.pid;

events {
    worker_connections $NGINX_WORKER_CONNECTIONS;
    use epoll;
    multi_accept on;
}

http {
    include       mime.types;
    default_type  application/octet-stream;
    
    # Logging format
    log_format main '\$remote_addr - \$remote_user [\$time_local] "\$request" '
                   '\$status \$body_bytes_sent "\$http_referer" '
                   '"\$http_user_agent" "\$http_x_forwarded_for"';
    
    # Basic settings
    sendfile on;
    tcp_nopush on;
    tcp_nodelay on;
    keepalive_timeout $NGINX_KEEPALIVE_TIMEOUT;
    types_hash_max_size 2048;
    server_tokens off;
    
    # Buffer settings
    client_body_buffer_size 128k;
    client_max_body_size $NGINX_CLIENT_MAX_BODY_SIZE;
    client_header_buffer_size 1k;
    large_client_header_buffers 4 4k;
    output_buffers 1 32k;
    postpone_output 1460;
    
    # Timeouts
    client_body_timeout 12;
    client_header_timeout 12;
    send_timeout 10;
    
    # Compression
    gzip on;
    gzip_vary on;
    gzip_min_length $GZIP_MIN_LENGTH;
    gzip_comp_level $GZIP_COMPRESSION_LEVEL;
    gzip_types
        text/plain
        text/css
        text/xml
        text/javascript
        application/json
        application/javascript
        application/xml+rss
        application/atom+xml
        image/svg+xml;
EOF

    # Add security headers if enabled
    if [[ "$SECURITY_HEADERS_ENABLED" == "true" ]]; then
        cat >> "$nginx_conf" << EOF
    
    # Security headers
    add_header X-Frame-Options DENY always;
    add_header X-Content-Type-Options nosniff always;
    add_header X-XSS-Protection "1; mode=block" always;
    add_header Referrer-Policy "strict-origin-when-cross-origin" always;
EOF
        
        if [[ "$HSTS_ENABLED" == "true" ]]; then
            cat >> "$nginx_conf" << EOF
    add_header Strict-Transport-Security "max-age=$HSTS_MAX_AGE; includeSubDomains; preload" always;
EOF
        fi
    fi
    
    # Add rate limiting if enabled
    if [[ "$RATE_LIMIT_ENABLED" == "true" ]]; then
        cat >> "$nginx_conf" << EOF
    
    # Rate limiting zones
    limit_req_zone \$binary_remote_addr zone=login:$RATE_LIMIT_ZONE_SIZE rate=$RATE_LIMIT_LOGIN;
    limit_req_zone \$binary_remote_addr zone=api:$RATE_LIMIT_ZONE_SIZE rate=$RATE_LIMIT_API;
    limit_req_zone \$binary_remote_addr zone=general:$RATE_LIMIT_ZONE_SIZE rate=$RATE_LIMIT_GENERAL;
EOF
    fi
    
    # Add caching if enabled
    if [[ "$NGINX_FASTCGI_CACHE" == "true" ]]; then
        cat >> "$nginx_conf" << EOF
    
    # FastCGI cache
    fastcgi_cache_path $CACHE_DIR/fastcgi levels=1:2 keys_zone=WORDPRESS:100m inactive=60m;
    fastcgi_cache_key "\$scheme\$request_method\$host\$request_uri";
    fastcgi_cache_use_stale error timeout invalid_header http_500;
EOF
    fi
    
    cat >> "$nginx_conf" << EOF
    
    # Include virtual hosts
    include $conf_dir/conf.d/*.conf;
}
EOF

    # Create log directory
    mkdir -p "$LOG_DIR/nginx"
    chown -R "$NGINX_USER:$NGINX_GROUP" "$LOG_DIR/nginx"
    
    log_message "SUCCESS" "Nginx configuration created"
}

# Create Nginx systemd service
create_nginx_service() {
    cat > /etc/systemd/system/nginx.service << 'EOF'
[Unit]
Description=The nginx HTTP and reverse proxy server
After=network.target remote-fs.target nss-lookup.target

[Service]
Type=forking
PIDFile=/var/run/nginx.pid
ExecStartPre=/usr/local/nginx/sbin/nginx -t
ExecStart=/usr/local/nginx/sbin/nginx
ExecReload=/bin/kill -s HUP $MAINPID
KillSignal=SIGQUIT
TimeoutStopSec=5
KillMode=mixed
PrivateTmp=true

[Install]
WantedBy=multi-user.target
EOF

    systemctl daemon-reload
    systemctl enable nginx
    
    log_message "SUCCESS" "Nginx systemd service created"
}

# Start Nginx service
start_nginx() {
    log_message "INFO" "Starting Nginx service..."
    
    if test_config "nginx" && systemctl start nginx; then
        log_message "SUCCESS" "Nginx started successfully"
        return 0
    else
        log_message "ERROR" "Failed to start Nginx"
        return 1
    fi
}

# Stop Nginx service
stop_nginx() {
    log_message "INFO" "Stopping Nginx service..."
    
    if systemctl stop nginx; then
        log_message "SUCCESS" "Nginx stopped successfully"
        return 0
    else
        log_message "ERROR" "Failed to stop Nginx"
        return 1
    fi
}

# Upgrade/downgrade Nginx
upgrade_nginx() {
    local new_version=$1
    
    if [[ -z "$new_version" ]]; then
        log_message "ERROR" "Please specify Nginx version to upgrade to"
        return 1
    fi
    
    log_message "INFO" "Upgrading Nginx to version $new_version..."
    
    # Stop current Nginx
    stop_nginx
    
    # Backup current installation
    if [[ -d "/usr/local/nginx" ]]; then
        mv "/usr/local/nginx" "/usr/local/nginx.backup.$(date +%Y%m%d_%H%M%S)"
    fi
    
    # Install new version
    if install_nginx "$new_version"; then
        log_message "SUCCESS" "Nginx upgraded to version $new_version successfully"
        start_nginx
        return 0
    else
        log_message "ERROR" "Failed to upgrade Nginx"
        return 1
    fi
}

# Get Nginx status
nginx_status() {
    echo "=== Nginx Status ==="
    echo "Version: $(nginx -v 2>&1 | cut -d'/' -f2)"
    echo "Status: $(systemctl is-active nginx)"
    echo "Configuration test: $(nginx -t 2>&1 | tail -1)"
    echo "Process ID: $(cat /var/run/nginx.pid 2>/dev/null || echo 'Not running')"
    echo "Workers: $(ps aux | grep '[n]ginx: worker' | wc -l)"
}

# Show Nginx configuration
show_nginx_config() {
    echo "=== Nginx Configuration ==="
    echo "Main config: /usr/local/nginx/conf/nginx.conf"
    echo "Virtual hosts: /usr/local/nginx/conf/conf.d/"
    echo "Log directory: $LOG_DIR/nginx/"
    echo "Cache directory: $CACHE_DIR/"
    echo ""
    echo "Active virtual hosts:"
    ls -la /usr/local/nginx/conf/conf.d/*.conf 2>/dev/null | awk '{print $9}' | xargs -I {} basename {} .conf
}

# Main execution
case "${1:-help}" in
    "install")
        install_nginx "$2"
        ;;
    "start")
        start_nginx
        ;;
    "stop")
        stop_nginx
        ;;
    "restart")
        stop_nginx && start_nginx
        ;;
    "upgrade")
        upgrade_nginx "$2"
        ;;
    "status")
        nginx_status
        ;;
    "config")
        show_nginx_config
        ;;
    "test")
        test_config "nginx"
        ;;
    *)
        echo "ThangLQ Stack - Nginx Module"
        echo "Usage: $0 <command> [options]"
        echo ""
        echo "Commands:"
        echo "  install [version]  - Install Nginx from source"
        echo "  start              - Start Nginx service"
        echo "  stop               - Stop Nginx service"
        echo "  restart            - Restart Nginx service"
        echo "  upgrade <version>  - Upgrade/downgrade Nginx"
        echo "  status             - Show Nginx status"
        echo "  config             - Show configuration info"
        echo "  test               - Test Nginx configuration"
        echo ""
        echo "Examples:"
        echo "  $0 install 1.24.0"
        echo "  $0 upgrade 1.25.0"
        echo "  $0 status"
        ;;
esac 