#!/bin/bash

# ================================================================
# ThangLQ Stack - MariaDB Module
# MariaDB installation and management
# ================================================================

# Source common utilities
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
source "$SCRIPT_DIR/../../utils/common.sh"

# Install MariaDB
install_mariadb() {
    local version=${1:-$MARIADB_VERSION}
    
    log_message "INFO" "Installing MariaDB $version..."
    
    # Add MariaDB repository
    if command -v dnf >/dev/null 2>&1; then
        # For RHEL 8/9, AlmaLinux, Rocky Linux
        cat > /etc/yum.repos.d/mariadb.repo << EOF
[mariadb]
name = MariaDB
baseurl = http://yum.mariadb.org/$version/rhel8-amd64/
gpgkey=https://yum.mariadb.org/RPM-GPG-KEY-MariaDB
gpgcheck=1
enabled=1
EOF
        dnf install -y MariaDB-server MariaDB-client
    else
        # For CentOS 7
        cat > /etc/yum.repos.d/mariadb.repo << EOF
[mariadb]
name = MariaDB
baseurl = http://yum.mariadb.org/$version/centos7-amd64/
gpgkey=https://yum.mariadb.org/RPM-GPG-KEY-MariaDB
gpgcheck=1
enabled=1
EOF
        yum install -y MariaDB-server MariaDB-client
    fi
    
    # Create MariaDB configuration
    create_mariadb_config
    
    # Enable and start MariaDB
    systemctl enable mariadb
    systemctl start mariadb
    
    # Secure installation
    secure_mariadb_installation
    
    log_message "SUCCESS" "MariaDB $version installed successfully"
    return 0
}

# Create MariaDB configuration
create_mariadb_config() {
    log_message "INFO" "Creating MariaDB configuration..."
    
    # Calculate innodb_buffer_pool_size based on available RAM
    get_system_info
    local buffer_pool_size
    
    if [[ $TOTAL_RAM -ge 8192 ]]; then
        buffer_pool_size="4G"
    elif [[ $TOTAL_RAM -ge 4096 ]]; then
        buffer_pool_size="2G"
    elif [[ $TOTAL_RAM -ge 2048 ]]; then
        buffer_pool_size="1G"
    else
        buffer_pool_size="512M"
    fi
    
    # Backup existing configuration
    backup_file "/etc/my.cnf.d/server.cnf"
    
    # Create optimized MariaDB configuration
    cat > /etc/my.cnf.d/server.cnf << EOF
[server]

[mysqld]
# Basic settings
bind-address = $MARIADB_BIND_ADDRESS
port = $MARIADB_PORT
datadir = /var/lib/mysql
socket = /var/lib/mysql/mysql.sock
tmpdir = /tmp
skip-external-locking

# Character set and collation
character-set-server = $MARIADB_CHARACTER_SET
collation-server = $MARIADB_COLLATION
init-connect = 'SET NAMES $MARIADB_CHARACTER_SET'

# Logging
log-error = /var/log/mariadb/mariadb.log
slow_query_log = 1
slow_query_log_file = /var/log/mariadb/slow.log
long_query_time = 2
log_queries_not_using_indexes = 1

# Performance settings
max_connections = 151
connect_timeout = 10
wait_timeout = 600
max_allowed_packet = 64M
thread_cache_size = 128
sort_buffer_size = 4M
bulk_insert_buffer_size = 16M
tmp_table_size = 32M
max_heap_table_size = 32M

# MyISAM settings
myisam_recover_options = BACKUP
key_buffer_size = 128M
table_open_cache = 400
myisam_sort_buffer_size = 512M
concurrent_insert = 2
read_buffer_size = 2M
read_rnd_buffer_size = 1M

# InnoDB settings
innodb_buffer_pool_size = $buffer_pool_size
innodb_log_file_size = 256M
innodb_log_buffer_size = 8M
innodb_flush_log_at_trx_commit = 1
innodb_lock_wait_timeout = 50
innodb_file_per_table = 1
innodb_flush_method = O_DIRECT

# Query cache
query_cache_type = 1
query_cache_limit = 2M
query_cache_size = 64M

[mysql]
default-character-set = $MARIADB_CHARACTER_SET

[mysqldump]
max_allowed_packet = 64M

[client]
port = $MARIADB_PORT
socket = /var/lib/mysql/mysql.sock
default-character-set = $MARIADB_CHARACTER_SET
EOF

    # Create log directory
    mkdir -p /var/log/mariadb
    chown mysql:mysql /var/log/mariadb
    
    log_message "SUCCESS" "MariaDB configuration created"
}

# Secure MariaDB installation
secure_mariadb_installation() {
    log_message "INFO" "Securing MariaDB installation..."
    
    # Generate root password if not set
    if [[ -z "$MARIADB_ROOT_PASSWORD" ]]; then
        MARIADB_ROOT_PASSWORD=$(generate_password 16)
        log_message "INFO" "Generated root password: $MARIADB_ROOT_PASSWORD"
        echo "MariaDB root password: $MARIADB_ROOT_PASSWORD" >> "$LOG_DIR/credentials.txt"
    fi
    
    # Set root password and secure installation
    mysql -e "UPDATE mysql.user SET Password=PASSWORD('$MARIADB_ROOT_PASSWORD') WHERE User='root';"
    mysql -e "DELETE FROM mysql.user WHERE User='';"
    mysql -e "DELETE FROM mysql.user WHERE User='root' AND Host NOT IN ('localhost', '127.0.0.1', '::1');"
    mysql -e "DROP DATABASE IF EXISTS test;"
    mysql -e "DELETE FROM mysql.db WHERE Db='test' OR Db='test\\_%';"
    mysql -e "FLUSH PRIVILEGES;"
    
    # Create .my.cnf for root
    cat > /root/.my.cnf << EOF
[client]
user=root
password=$MARIADB_ROOT_PASSWORD
EOF
    chmod 600 /root/.my.cnf
    
    log_message "SUCCESS" "MariaDB secured successfully"
}

# Start MariaDB service
start_mariadb() {
    log_message "INFO" "Starting MariaDB service..."
    
    if systemctl start mariadb; then
        log_message "SUCCESS" "MariaDB started successfully"
        return 0
    else
        log_message "ERROR" "Failed to start MariaDB"
        return 1
    fi
}

# Stop MariaDB service
stop_mariadb() {
    log_message "INFO" "Stopping MariaDB service..."
    
    if systemctl stop mariadb; then
        log_message "SUCCESS" "MariaDB stopped successfully"
        return 0
    else
        log_message "ERROR" "Failed to stop MariaDB"
        return 1
    fi
}

# Get MariaDB status
mariadb_status() {
    echo "=== MariaDB Status ==="
    echo "Version: $(mysql --version 2>/dev/null | awk '{print $5}' | cut -d',' -f1 || echo 'Not available')"
    echo "Status: $(systemctl is-active mariadb 2>/dev/null || echo 'Not installed')"
    echo "Port: $MARIADB_PORT"
    echo "Bind Address: $MARIADB_BIND_ADDRESS"
    echo "Data Directory: /var/lib/mysql"
    echo "Log Directory: /var/log/mariadb"
    
    # Show running processes
    if systemctl is-active mariadb >/dev/null 2>&1; then
        echo ""
        echo "Database Status:"
        mysql -e "SHOW STATUS LIKE 'Uptime%'; SHOW STATUS LIKE 'Threads_connected'; SHOW STATUS LIKE 'Questions';" 2>/dev/null || echo "Cannot connect to database"
    fi
}

# Test MariaDB connection
test_mariadb() {
    log_message "INFO" "Testing MariaDB connection..."
    
    if mysql -e "SELECT 1;" >/dev/null 2>&1; then
        log_message "SUCCESS" "MariaDB connection test passed"
        
        # Show databases
        echo "Available databases:"
        mysql -e "SHOW DATABASES;" 2>/dev/null || echo "Cannot list databases"
        
        return 0
    else
        log_message "ERROR" "Cannot connect to MariaDB"
        return 1
    fi
}

# Create database
create_database() {
    local db_name=$1
    local db_user=$2
    local db_password=$3
    
    if [[ -z "$db_name" ]]; then
        log_message "ERROR" "Database name required"
        return 1
    fi
    
    log_message "INFO" "Creating database: $db_name"
    
    # Generate password if not provided
    if [[ -z "$db_password" ]]; then
        db_password=$(generate_password 12)
    fi
    
    # Create database and user
    mysql -e "CREATE DATABASE IF NOT EXISTS \`$db_name\` CHARACTER SET $MARIADB_CHARACTER_SET COLLATE $MARIADB_COLLATION;"
    
    if [[ -n "$db_user" ]]; then
        mysql -e "CREATE USER IF NOT EXISTS '$db_user'@'localhost' IDENTIFIED BY '$db_password';"
        mysql -e "GRANT ALL PRIVILEGES ON \`$db_name\`.* TO '$db_user'@'localhost';"
        mysql -e "FLUSH PRIVILEGES;"
        
        log_message "SUCCESS" "Database $db_name created with user $db_user"
        echo "Database: $db_name, User: $db_user, Password: $db_password" >> "$LOG_DIR/credentials.txt"
    else
        log_message "SUCCESS" "Database $db_name created"
    fi
    
    return 0
}

# Show MariaDB configuration
show_mariadb_config() {
    echo "=== MariaDB Configuration ==="
    echo "Config file: /etc/my.cnf.d/server.cnf"
    echo "Data directory: /var/lib/mysql"
    echo "Log directory: /var/log/mariadb"
    echo "Socket: /var/lib/mysql/mysql.sock"
    echo ""
    echo "Current configuration:"
    if [[ -f "/etc/my.cnf.d/server.cnf" ]]; then
        cat "/etc/my.cnf.d/server.cnf"
    else
        echo "Configuration file not found"
    fi
}

# Main execution
case "${1:-help}" in
    "install")
        install_mariadb "$2"
        ;;
    "start")
        start_mariadb
        ;;
    "stop")
        stop_mariadb
        ;;
    "restart")
        stop_mariadb && start_mariadb
        ;;
    "status")
        mariadb_status
        ;;
    "test")
        test_mariadb
        ;;
    "create-db")
        create_database "$2" "$3" "$4"
        ;;
    "config")
        show_mariadb_config
        ;;
    *)
        echo "ThangLQ Stack - MariaDB Module"
        echo "Usage: $0 <command> [options]"
        echo ""
        echo "Commands:"
        echo "  install [version]        - Install MariaDB"
        echo "  start                    - Start MariaDB service"
        echo "  stop                     - Stop MariaDB service"
        echo "  restart                  - Restart MariaDB service"
        echo "  status                   - Show MariaDB status"
        echo "  test                     - Test MariaDB connection"
        echo "  create-db <name> [user] [password] - Create database"
        echo "  config                   - Show configuration"
        echo ""
        echo "Examples:"
        echo "  $0 install 10.11"
        echo "  $0 create-db myapp myuser"
        echo "  $0 status"
        ;;
esac 