#!/bin/bash

# ================================================================
# ThangLQ Stack - OpenResty Module
# OpenResty Nginx with Lua support installation and management
# ================================================================

# Source common utilities
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
source "$SCRIPT_DIR/../../utils/common.sh"

# Install OpenResty
install_openresty() {
    local version=${1:-$OPENRESTY_VERSION}
    
    log_message "INFO" "Installing OpenResty $version from source..."
    
    cd /tmp
    
    # Download OpenResty source
    local openresty_url="https://openresty.org/download/openresty-${version}.tar.gz"
    wget "$openresty_url" -O "openresty-${version}.tar.gz"
    
    if [[ ! -f "openresty-${version}.tar.gz" ]]; then
        log_message "ERROR" "Failed to download OpenResty"
        return 1
    fi
    
    tar -xzf "openresty-${version}.tar.gz"
    cd "openresty-${version}"
    
    # Configure OpenResty with Lua modules
    ./configure \
        --prefix=/usr/local/openresty \
        --user=$NGINX_USER \
        --group=$NGINX_GROUP \
        --with-http_ssl_module \
        --with-http_v2_module \
        --with-http_realip_module \
        --with-http_addition_module \
        --with-http_sub_module \
        --with-http_dav_module \
        --with-http_flv_module \
        --with-http_mp4_module \
        --with-http_gunzip_module \
        --with-http_gzip_static_module \
        --with-http_random_index_module \
        --with-http_secure_link_module \
        --with-http_stub_status_module \
        --with-http_auth_request_module \
        --with-http_xslt_module=dynamic \
        --with-http_image_filter_module=dynamic \
        --with-http_geoip_module=dynamic \
        --with-threads \
        --with-stream \
        --with-stream_ssl_module \
        --with-stream_ssl_preread_module \
        --with-stream_realip_module \
        --with-stream_geoip_module=dynamic \
        --with-http_slice_module \
        --with-mail \
        --with-mail_ssl_module \
        --with-compat \
        --with-file-aio \
        --with-luajit \
        $OPENRESTY_EXTRA_MODULES \
        $NGINX_CONFIGURE_ARGS
    
    if [[ $? -ne 0 ]]; then
        log_message "ERROR" "OpenResty configure failed"
        return 1
    fi
    
    # Compile OpenResty
    make -j$(nproc)
    if [[ $? -ne 0 ]]; then
        log_message "ERROR" "OpenResty compilation failed"
        return 1
    fi
    
    # Install OpenResty
    make install
    if [[ $? -ne 0 ]]; then
        log_message "ERROR" "OpenResty installation failed"
        return 1
    fi
    
    # Create symlinks
    ln -sf /usr/local/openresty/nginx/sbin/nginx /usr/local/bin/nginx
    ln -sf /usr/local/openresty/bin/openresty /usr/local/bin/openresty
    ln -sf /usr/local/openresty/bin/resty /usr/local/bin/resty
    
    # Create nginx user
    create_user "$NGINX_USER"
    
    # Create OpenResty configuration
    create_openresty_config
    
    # Create systemd service
    create_openresty_service
    
    # Install Lua rocks if needed
    install_lua_rocks
    
    log_message "SUCCESS" "OpenResty $version installed successfully"
    return 0
}

# Create OpenResty configuration
create_openresty_config() {
    local nginx_conf="/usr/local/openresty/nginx/conf/nginx.conf"
    local conf_dir="/usr/local/openresty/nginx/conf"
    
    # Create directories
    mkdir -p "$conf_dir"/{conf.d,ssl,includes,lua}
    mkdir -p "$CACHE_DIR"/{fastcgi,proxy,scgi,uwsgi}
    
    # Backup existing configuration
    backup_file "$nginx_conf"
    
    # Create main configuration with Lua support
    cat > "$nginx_conf" << EOF
user $NGINX_USER;
worker_processes $NGINX_WORKER_PROCESSES;
worker_rlimit_nofile 65535;
error_log $LOG_DIR/nginx/error.log $NGINX_ERROR_LOG_LEVEL;
pid /var/run/nginx.pid;

# Lua package path
env LUA_PATH;
env LUA_CPATH;

events {
    worker_connections $NGINX_WORKER_CONNECTIONS;
    use epoll;
    multi_accept on;
}

http {
    include       mime.types;
    default_type  application/octet-stream;
    
    # Lua shared dictionaries
    lua_shared_dict cache 100m;
    lua_shared_dict locks 1m;
    
    # Lua code cache
    lua_code_cache on;
    
    # Logging format
    log_format main '\$remote_addr - \$remote_user [\$time_local] "\$request" '
                   '\$status \$body_bytes_sent "\$http_referer" '
                   '"\$http_user_agent" "\$http_x_forwarded_for"';
    
    # Basic settings
    sendfile on;
    tcp_nopush on;
    tcp_nodelay on;
    keepalive_timeout $NGINX_KEEPALIVE_TIMEOUT;
    types_hash_max_size 2048;
    server_tokens off;
    
    # Buffer settings
    client_body_buffer_size 128k;
    client_max_body_size $NGINX_CLIENT_MAX_BODY_SIZE;
    client_header_buffer_size 1k;
    large_client_header_buffers 4 4k;
    output_buffers 1 32k;
    postpone_output 1460;
    
    # Timeouts
    client_body_timeout 12;
    client_header_timeout 12;
    send_timeout 10;
    
    # Compression
    gzip on;
    gzip_vary on;
    gzip_min_length $GZIP_MIN_LENGTH;
    gzip_comp_level $GZIP_COMPRESSION_LEVEL;
    gzip_types
        text/plain
        text/css
        text/xml
        text/javascript
        application/json
        application/javascript
        application/xml+rss
        application/atom+xml
        image/svg+xml;
EOF

    # Add security headers if enabled
    if [[ "$SECURITY_HEADERS_ENABLED" == "true" ]]; then
        cat >> "$nginx_conf" << EOF
    
    # Security headers
    add_header X-Frame-Options DENY always;
    add_header X-Content-Type-Options nosniff always;
    add_header X-XSS-Protection "1; mode=block" always;
    add_header Referrer-Policy "strict-origin-when-cross-origin" always;
EOF
        
        if [[ "$HSTS_ENABLED" == "true" ]]; then
            cat >> "$nginx_conf" << EOF
    add_header Strict-Transport-Security "max-age=$HSTS_MAX_AGE; includeSubDomains; preload" always;
EOF
        fi
    fi
    
    # Add rate limiting if enabled
    if [[ "$RATE_LIMIT_ENABLED" == "true" ]]; then
        cat >> "$nginx_conf" << EOF
    
    # Rate limiting zones
    limit_req_zone \$binary_remote_addr zone=login:$RATE_LIMIT_ZONE_SIZE rate=$RATE_LIMIT_LOGIN;
    limit_req_zone \$binary_remote_addr zone=api:$RATE_LIMIT_ZONE_SIZE rate=$RATE_LIMIT_API;
    limit_req_zone \$binary_remote_addr zone=general:$RATE_LIMIT_ZONE_SIZE rate=$RATE_LIMIT_GENERAL;
EOF
    fi
    
    # Add caching if enabled
    if [[ "$NGINX_FASTCGI_CACHE" == "true" ]]; then
        cat >> "$nginx_conf" << EOF
    
    # FastCGI cache
    fastcgi_cache_path $CACHE_DIR/fastcgi levels=1:2 keys_zone=WORDPRESS:100m inactive=60m;
    fastcgi_cache_key "\$scheme\$request_method\$host\$request_uri";
    fastcgi_cache_use_stale error timeout invalid_header http_500;
EOF
    fi
    
    cat >> "$nginx_conf" << EOF
    
    # Include virtual hosts
    include $conf_dir/conf.d/*.conf;
}
EOF

    # Create basic Lua scripts directory
    create_lua_scripts
    
    # Create log directory
    mkdir -p "$LOG_DIR/nginx"
    chown -R "$NGINX_USER:$NGINX_GROUP" "$LOG_DIR/nginx"
    
    log_message "SUCCESS" "OpenResty configuration created"
}

# Create basic Lua scripts
create_lua_scripts() {
    local lua_dir="/usr/local/openresty/nginx/conf/lua"
    
    # Create a sample Lua script for security
    cat > "$lua_dir/security.lua" << 'EOF'
-- ThangLQ Stack - Basic Security Lua Script
local ngx = ngx
local string = string

-- Block common attack patterns
local uri = ngx.var.uri
local user_agent = ngx.var.http_user_agent or ""

-- Block common SQL injection patterns
if string.match(uri, "union.*select") or 
   string.match(uri, "drop.*table") or
   string.match(uri, "insert.*into") then
    ngx.log(ngx.ERR, "SQL injection attempt detected from " .. ngx.var.remote_addr)
    ngx.status = 403
    ngx.say("Access denied")
    ngx.exit(403)
end

-- Block common XSS patterns
if string.match(uri, "<script") or 
   string.match(uri, "javascript:") then
    ngx.log(ngx.ERR, "XSS attempt detected from " .. ngx.var.remote_addr)
    ngx.status = 403
    ngx.say("Access denied")
    ngx.exit(403)
end

-- Block bad user agents
if string.match(user_agent, "sqlmap") or
   string.match(user_agent, "nikto") or
   string.match(user_agent, "nmap") then
    ngx.log(ngx.ERR, "Bad user agent detected: " .. user_agent)
    ngx.status = 403
    ngx.say("Access denied")
    ngx.exit(403)
end
EOF

    # Create a sample Lua script for caching
    cat > "$lua_dir/cache.lua" << 'EOF'
-- ThangLQ Stack - Simple Caching Lua Script
local cache = ngx.shared.cache
local key = ngx.var.scheme .. ngx.var.host .. ngx.var.request_uri

-- Try to get from cache
local cached_content = cache:get(key)
if cached_content then
    ngx.header["X-Cache"] = "HIT"
    ngx.say(cached_content)
    ngx.exit(200)
end

-- If not in cache, mark as MISS
ngx.header["X-Cache"] = "MISS"
EOF

    log_message "SUCCESS" "Basic Lua scripts created"
}

# Install Lua rocks (package manager)
install_lua_rocks() {
    log_message "INFO" "Installing LuaRocks..."
    
    cd /tmp
    
    # Download LuaRocks
    wget https://luarocks.org/releases/luarocks-3.9.2.tar.gz
    tar -xzf luarocks-3.9.2.tar.gz
    cd luarocks-3.9.2
    
    # Configure and install
    ./configure --prefix=/usr/local/openresty/luajit \
                --with-lua=/usr/local/openresty/luajit/ \
                --lua-suffix=jit \
                --with-lua-include=/usr/local/openresty/luajit/include/luajit-2.1
    
    make && make install
    
    # Create symlink
    ln -sf /usr/local/openresty/luajit/bin/luarocks /usr/local/bin/luarocks
    
    log_message "SUCCESS" "LuaRocks installed"
}

# Create OpenResty systemd service
create_openresty_service() {
    cat > /etc/systemd/system/openresty.service << 'EOF'
[Unit]
Description=The OpenResty HTTP and reverse proxy server
After=network.target remote-fs.target nss-lookup.target

[Service]
Type=forking
PIDFile=/var/run/nginx.pid
ExecStartPre=/usr/local/openresty/nginx/sbin/nginx -t
ExecStart=/usr/local/openresty/nginx/sbin/nginx
ExecReload=/bin/kill -s HUP $MAINPID
KillSignal=SIGQUIT
TimeoutStopSec=5
KillMode=mixed
PrivateTmp=true

[Install]
WantedBy=multi-user.target
EOF

    # Also create nginx alias service
    cat > /etc/systemd/system/nginx.service << 'EOF'
[Unit]
Description=The OpenResty HTTP and reverse proxy server (nginx alias)
After=network.target remote-fs.target nss-lookup.target

[Service]
Type=forking
PIDFile=/var/run/nginx.pid
ExecStartPre=/usr/local/openresty/nginx/sbin/nginx -t
ExecStart=/usr/local/openresty/nginx/sbin/nginx
ExecReload=/bin/kill -s HUP $MAINPID
KillSignal=SIGQUIT
TimeoutStopSec=5
KillMode=mixed
PrivateTmp=true

[Install]
WantedBy=multi-user.target
EOF

    systemctl daemon-reload
    systemctl enable openresty
    
    log_message "SUCCESS" "OpenResty systemd service created"
}

# Start OpenResty service
start_openresty() {
    log_message "INFO" "Starting OpenResty service..."
    
    if test_config "nginx" && systemctl start openresty; then
        log_message "SUCCESS" "OpenResty started successfully"
        return 0
    else
        log_message "ERROR" "Failed to start OpenResty"
        return 1
    fi
}

# Stop OpenResty service
stop_openresty() {
    log_message "INFO" "Stopping OpenResty service..."
    
    if systemctl stop openresty; then
        log_message "SUCCESS" "OpenResty stopped successfully"
        return 0
    else
        log_message "ERROR" "Failed to stop OpenResty"
        return 1
    fi
}

# Upgrade/downgrade OpenResty
upgrade_openresty() {
    local new_version=$1
    
    if [[ -z "$new_version" ]]; then
        log_message "ERROR" "Please specify OpenResty version to upgrade to"
        return 1
    fi
    
    log_message "INFO" "Upgrading OpenResty to version $new_version..."
    
    # Stop current OpenResty
    stop_openresty
    
    # Backup current installation
    if [[ -d "/usr/local/openresty" ]]; then
        mv "/usr/local/openresty" "/usr/local/openresty.backup.$(date +%Y%m%d_%H%M%S)"
    fi
    
    # Install new version
    if install_openresty "$new_version"; then
        log_message "SUCCESS" "OpenResty upgraded to version $new_version successfully"
        start_openresty
        return 0
    else
        log_message "ERROR" "Failed to upgrade OpenResty"
        return 1
    fi
}

# Get OpenResty status
openresty_status() {
    echo "=== OpenResty Status ==="
    echo "Version: $(nginx -v 2>&1 | cut -d'/' -f2)"
    echo "OpenResty Version: $(openresty -V 2>&1 | head -1 | cut -d'/' -f2)"
    echo "LuaJIT Version: $(resty -e 'print(jit.version)' 2>/dev/null || echo 'Not available')"
    echo "Status: $(systemctl is-active openresty)"
    echo "Configuration test: $(nginx -t 2>&1 | tail -1)"
    echo "Process ID: $(cat /var/run/nginx.pid 2>/dev/null || echo 'Not running')"
    echo "Workers: $(ps aux | grep '[n]ginx: worker' | wc -l)"
    echo "Lua Modules:"
    ls -la /usr/local/openresty/nginx/conf/lua/ 2>/dev/null || echo "  No Lua modules found"
}

# Show OpenResty configuration
show_openresty_config() {
    echo "=== OpenResty Configuration ==="
    echo "Main config: /usr/local/openresty/nginx/conf/nginx.conf"
    echo "Virtual hosts: /usr/local/openresty/nginx/conf/conf.d/"
    echo "Lua scripts: /usr/local/openresty/nginx/conf/lua/"
    echo "Log directory: $LOG_DIR/nginx/"
    echo "Cache directory: $CACHE_DIR/"
    echo ""
    echo "Active virtual hosts:"
    ls -la /usr/local/openresty/nginx/conf/conf.d/*.conf 2>/dev/null | awk '{print $9}' | xargs -I {} basename {} .conf
}

# Test Lua functionality
test_lua() {
    log_message "INFO" "Testing Lua functionality..."
    
    # Test LuaJIT
    if command -v resty >/dev/null 2>&1; then
        echo "LuaJIT Test:"
        resty -e 'print("Hello from LuaJIT " .. jit.version)'
        
        echo "Lua Libraries Test:"
        resty -e 'print("Math: " .. math.pi)'
        resty -e 'print("String: " .. string.upper("hello"))'
        
        log_message "SUCCESS" "Lua functionality test completed"
    else
        log_message "ERROR" "Resty command not found"
        return 1
    fi
}

# Install common Lua modules
install_lua_modules() {
    log_message "INFO" "Installing common Lua modules..."
    
    if command -v luarocks >/dev/null 2>&1; then
        # Install common modules
        luarocks install lua-resty-http
        luarocks install lua-resty-redis
        luarocks install lua-resty-mysql
        luarocks install lua-resty-memcached
        luarocks install lua-cjson
        
        log_message "SUCCESS" "Common Lua modules installed"
    else
        log_message "WARNING" "LuaRocks not available, skipping module installation"
    fi
}

# Main execution
case "${1:-help}" in
    "install")
        install_openresty "$2"
        ;;
    "start")
        start_openresty
        ;;
    "stop")
        stop_openresty
        ;;
    "restart")
        stop_openresty && start_openresty
        ;;
    "upgrade")
        upgrade_openresty "$2"
        ;;
    "status")
        openresty_status
        ;;
    "config")
        show_openresty_config
        ;;
    "test")
        test_config "nginx"
        ;;
    "test-lua")
        test_lua
        ;;
    "install-modules")
        install_lua_modules
        ;;
    *)
        echo "ThangLQ Stack - OpenResty Module"
        echo "Usage: $0 <command> [options]"
        echo ""
        echo "Commands:"
        echo "  install [version]   - Install OpenResty from source"
        echo "  start               - Start OpenResty service"
        echo "  stop                - Stop OpenResty service"
        echo "  restart             - Restart OpenResty service"
        echo "  upgrade <version>   - Upgrade/downgrade OpenResty"
        echo "  status              - Show OpenResty status"
        echo "  config              - Show configuration info"
        echo "  test                - Test OpenResty configuration"
        echo "  test-lua            - Test Lua functionality"
        echo "  install-modules     - Install common Lua modules"
        echo ""
        echo "Examples:"
        echo "  $0 install 1.25.3.1"
        echo "  $0 upgrade 1.25.3.2"
        echo "  $0 test-lua"
        echo "  $0 install-modules"
        ;;
esac 