#!/bin/bash

# ================================================================
# ThangLQ Stack - PHP-FPM Module
# Multi-version PHP-FPM installation and management
# ================================================================

# Source common utilities
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
source "$SCRIPT_DIR/../../utils/common.sh"

# Install PHP-FPM
install_php_fpm() {
    local version=${1:-$PHP_DEFAULT_VERSION}
    
    if ! validate_php_version "$version"; then
        return 1
    fi
    
    log_message "INFO" "Installing PHP-FPM $version from source..."
    
    # Install dependencies first
    install_php_dependencies
    
    cd /tmp
    
    # Download PHP source
    local php_url="https://www.php.net/distributions/php-${version}.tar.gz"
    wget "$php_url" -O "php-${version}.tar.gz"
    
    if [[ ! -f "php-${version}.tar.gz" ]]; then
        log_message "ERROR" "Failed to download PHP $version"
        return 1
    fi
    
    tar -xzf "php-${version}.tar.gz"
    cd "php-${version}"
    
    # Get major.minor version for path
    local php_major_minor=$(echo "$version" | cut -d. -f1-2)
    local install_prefix="/usr/local/php${php_major_minor}"
    
    # Configure PHP with extensive options
    ./configure \
        --prefix="$install_prefix" \
        --with-config-file-path="$install_prefix/etc" \
        --with-config-file-scan-dir="$install_prefix/etc/conf.d" \
        --with-fpm-user="$PHP_FPM_USER" \
        --with-fpm-group="$PHP_FPM_GROUP" \
        $PHP_CONFIGURE_ARGS
    
    if [[ $? -ne 0 ]]; then
        log_message "ERROR" "PHP configure failed"
        return 1
    fi
    
    # Compile PHP
    make -j$(nproc)
    if [[ $? -ne 0 ]]; then
        log_message "ERROR" "PHP compilation failed"
        return 1
    fi
    
    # Install PHP
    make install
    if [[ $? -ne 0 ]]; then
        log_message "ERROR" "PHP installation failed"
        return 1
    fi
    
    # Create symlinks for default version
    if [[ "$version" == "$PHP_DEFAULT_VERSION" ]]; then
        ln -sf "$install_prefix/bin/php" /usr/local/bin/php
        ln -sf "$install_prefix/sbin/php-fpm" /usr/local/bin/php-fpm
        ln -sf "$install_prefix/bin/phpize" /usr/local/bin/phpize
        ln -sf "$install_prefix/bin/php-config" /usr/local/bin/php-config
    fi
    
    # Create PHP configuration
    create_php_config "$php_major_minor"
    
    # Create PHP-FPM pool configuration
    create_php_fpm_config "$php_major_minor"
    
    # Create systemd service
    create_php_fpm_service "$php_major_minor"
    
    log_message "SUCCESS" "PHP-FPM $version installed successfully"
    return 0
}

# Install PHP dependencies
install_php_dependencies() {
    log_message "INFO" "Installing PHP dependencies..."
    
    if command -v dnf >/dev/null 2>&1; then
        dnf install -y \
            libxml2-devel sqlite-devel libcurl-devel libpng-devel \
            libjpeg-devel libwebp-devel freetype-devel libicu-devel \
            libxslt-devel readline-devel libzip-devel bzip2-devel \
            gmp-devel enchant-devel libsodium-devel oniguruma-devel \
            libedit-devel
    else
        yum install -y \
            libxml2-devel sqlite-devel libcurl-devel libpng-devel \
            libjpeg-devel libwebp-devel freetype-devel libicu-devel \
            libxslt-devel readline-devel libzip-devel bzip2-devel \
            gmp-devel enchant-devel libsodium-devel oniguruma-devel \
            libedit-devel
    fi
    
    log_message "SUCCESS" "PHP dependencies installed"
}

# Create PHP configuration
create_php_config() {
    local php_version=$1
    local php_ini="/usr/local/php${php_version}/etc/php.ini"
    local conf_dir="/usr/local/php${php_version}/etc/conf.d"
    
    # Create directories
    mkdir -p "$conf_dir"
    mkdir -p "/var/lib/php${php_version}/session"
    
    # Backup existing configuration
    backup_file "$php_ini"
    
    # Create main PHP configuration
    cat > "$php_ini" << EOF
[PHP]
; ThangLQ Stack PHP Configuration for PHP $php_version
; Generated on $(date)

; Engine
engine = On
short_open_tag = Off
precision = 14
output_buffering = 4096
zlib.output_compression = Off
implicit_flush = Off
unserialize_callback_func =
serialize_precision = -1
disable_functions = exec,passthru,shell_exec,system,proc_open,popen,curl_exec,curl_multi_exec,parse_ini_file,show_source
disable_classes =
zend.enable_gc = On

; Resource Limits
max_execution_time = 300
max_input_time = 300
memory_limit = 256M
post_max_size = 100M
upload_max_filesize = 100M
max_file_uploads = 20

; Error handling
error_reporting = E_ALL & ~E_DEPRECATED & ~E_STRICT
display_errors = Off
display_startup_errors = Off
log_errors = On
log_errors_max_len = 1024
ignore_repeated_errors = Off
ignore_repeated_source = Off
report_memleaks = On
track_errors = Off
html_errors = On
error_log = $LOG_DIR/php${php_version}/error.log

; Data Handling
variables_order = "GPCS"
request_order = "GP"
register_argc_argv = Off
auto_globals_jit = On
auto_prepend_file =
auto_append_file =
default_mimetype = "text/html"
default_charset = "UTF-8"

; File Uploads
file_uploads = On
upload_tmp_dir = /tmp
max_file_uploads = 20

; Fopen wrappers
allow_url_fopen = On
allow_url_include = Off
default_socket_timeout = 60
user_agent="PHP"

; Dynamic Extensions
extension_dir = "/usr/local/php${php_version}/lib/php/extensions/no-debug-non-zts-$(php-config --extension-dir | grep -o '[0-9]*$')"

; Module Settings

[CLI Server]
cli_server.color = On

[Date]
date.timezone = Asia/Ho_Chi_Minh

[Pdo_mysql]
pdo_mysql.cache_size = 2000
pdo_mysql.default_socket =

[mail function]
SMTP = localhost
smtp_port = 25
mail.add_x_header = On

[SQL]
sql.safe_mode = Off

[ODBC]
odbc.allow_persistent = On
odbc.check_persistent = On
odbc.max_persistent = -1
odbc.max_links = -1
odbc.defaultlrl = 4096
odbc.defaultbinmode = 1

[MySQL]
mysql.allow_local_infile = On
mysql.allow_persistent = On
mysql.cache_size = 2000
mysql.max_persistent = -1
mysql.max_links = -1
mysql.default_port =
mysql.default_socket =
mysql.default_host =
mysql.default_user =
mysql.default_password =
mysql.connect_timeout = 60
mysql.trace_mode = Off

[MySQLi]
mysqli.max_persistent = -1
mysqli.allow_persistent = On
mysqli.max_links = -1
mysqli.cache_size = 2000
mysqli.default_port = 3306
mysqli.default_socket =
mysqli.default_host =
mysqli.default_user =
mysqli.default_pw =
mysqli.reconnect = Off

[Session]
session.save_handler = files
session.save_path = "/var/lib/php${php_version}/session"
session.use_strict_mode = 0
session.use_cookies = 1
session.use_only_cookies = 1
session.name = PHPSESSID
session.auto_start = 0
session.cookie_lifetime = 0
session.cookie_path = /
session.cookie_domain =
session.cookie_httponly =
session.serialize_handler = php
session.gc_probability = 0
session.gc_divisor = 1000
session.gc_maxlifetime = 1440
session.referer_check =
session.cache_limiter = nocache
session.cache_expire = 180
session.use_trans_sid = 0
session.hash_function = 0
session.hash_bits_per_character = 5
session.upload_progress.enabled = On
session.upload_progress.cleanup = On
session.upload_progress.prefix = "upload_progress_"
session.upload_progress.name = "PHP_SESSION_UPLOAD_PROGRESS"
session.upload_progress.freq = "1%"
session.upload_progress.min_freq = "1"

[Assertion]
assert.active = 1
assert.warning = 1
assert.bail = 0
assert.callback = 0
assert.quiet_eval = 0

[mbstring]
mbstring.language = English
mbstring.internal_encoding =
mbstring.http_input =
mbstring.http_output =
mbstring.encoding_translation = Off
mbstring.detect_order = auto
mbstring.substitute_character = none
mbstring.func_overload = 0
mbstring.strict_detection = On
mbstring.http_output_conv_mimetype=

[gd]
gd.jpeg_ignore_warning = 0

[exif]
exif.encode_unicode = ISO-8859-15
exif.decode_unicode_motorola = UCS-2BE
exif.decode_unicode_intel = UCS-2LE
exif.encode_jis =
exif.decode_jis_motorola = JIS
exif.decode_jis_intel = JIS

[soap]
soap.wsdl_cache_enabled = 1
soap.wsdl_cache_dir = "/tmp"
soap.wsdl_cache_ttl = 86400
soap.wsdl_cache_limit = 5

[sysvshm]
sysvshm.init_mem = 10000

[ldap]
ldap.max_links = -1

[opcache]
opcache.enable = 1
opcache.enable_cli = 1
opcache.memory_consumption = 128
opcache.interned_strings_buffer = 8
opcache.max_accelerated_files = 4000
opcache.revalidate_freq = 2
opcache.fast_shutdown = 1
opcache.enable_file_override = 0
EOF

    # Create log directory
    mkdir -p "$LOG_DIR/php${php_version}"
    chown -R "$PHP_FPM_USER:$PHP_FPM_GROUP" "$LOG_DIR/php${php_version}"
    
    # Set session directory permissions
    chown -R "$PHP_FPM_USER:$PHP_FPM_GROUP" "/var/lib/php${php_version}/session"
    chmod 770 "/var/lib/php${php_version}/session"
    
    log_message "SUCCESS" "PHP $php_version configuration created"
}

# Create PHP-FPM pool configuration
create_php_fpm_config() {
    local php_version=$1
    local fpm_conf="/usr/local/php${php_version}/etc/php-fpm.conf"
    local pool_conf="/usr/local/php${php_version}/etc/php-fpm.d/www.conf"
    
    # Create pool directory
    mkdir -p "/usr/local/php${php_version}/etc/php-fpm.d"
    
    # Create main FPM configuration
    cat > "$fpm_conf" << EOF
[global]
pid = /var/run/php${php_version}-fpm.pid
error_log = $LOG_DIR/php${php_version}/fpm-error.log
log_level = notice
emergency_restart_threshold = 10
emergency_restart_interval = 1m
process_control_timeout = 10s
daemonize = yes
systemd_interval = 10

include=/usr/local/php${php_version}/etc/php-fpm.d/*.conf
EOF

    # Calculate pool settings based on system resources
    get_system_info
    local max_children
    local start_servers
    local min_spare_servers
    local max_spare_servers
    
    if [[ $TOTAL_RAM -ge 8192 ]]; then
        max_children=50
        start_servers=10
        min_spare_servers=5
        max_spare_servers=15
    elif [[ $TOTAL_RAM -ge 4096 ]]; then
        max_children=30
        start_servers=6
        min_spare_servers=3
        max_spare_servers=9
    elif [[ $TOTAL_RAM -ge 2048 ]]; then
        max_children=20
        start_servers=4
        min_spare_servers=2
        max_spare_servers=6
    else
        max_children=10
        start_servers=2
        min_spare_servers=1
        max_spare_servers=3
    fi

    # Create pool configuration
    cat > "$pool_conf" << EOF
[www]
user = $PHP_FPM_USER
group = $PHP_FPM_GROUP

listen = $PHP_FPM_SOCKET_DIR/php${php_version}-fpm.sock
listen.owner = $PHP_FPM_USER
listen.group = $PHP_FPM_GROUP
listen.mode = 0660

pm = dynamic
pm.max_children = $max_children
pm.start_servers = $start_servers
pm.min_spare_servers = $min_spare_servers
pm.max_spare_servers = $max_spare_servers
pm.process_idle_timeout = 10s
pm.max_requests = 1000

pm.status_path = /status
ping.path = /ping

access.log = $LOG_DIR/php${php_version}/access.log
access.format = "%R - %u %t \"%m %r%Q%q\" %s %f %{mili}d %{kilo}M %C%%"

slowlog = $LOG_DIR/php${php_version}/slow.log
request_slowlog_timeout = 5s

security.limit_extensions = .php .phar

env[HOSTNAME] = \$HOSTNAME
env[PATH] = /usr/local/bin:/usr/bin:/bin
env[TMP] = /tmp
env[TMPDIR] = /tmp
env[TEMP] = /tmp

php_admin_value[sendmail_path] = /usr/sbin/sendmail -t -i -f www@localhost
php_flag[display_errors] = off
php_admin_value[error_log] = $LOG_DIR/php${php_version}/error.log
php_admin_flag[log_errors] = on
php_admin_value[memory_limit] = 256M
php_admin_value[max_execution_time] = 300
php_admin_value[max_input_time] = 300
php_admin_value[max_input_vars] = 3000
php_admin_value[post_max_size] = 100M
php_admin_value[upload_max_filesize] = 100M
php_admin_value[session.save_handler] = files
php_admin_value[session.save_path] = /var/lib/php${php_version}/session
EOF

    log_message "SUCCESS" "PHP-FPM $php_version pool configuration created"
}

# Create PHP-FPM systemd service
create_php_fpm_service() {
    local php_version=$1
    
    cat > "/etc/systemd/system/php${php_version}-fpm.service" << EOF
[Unit]
Description=The PHP $php_version FastCGI Process Manager
After=network.target

[Service]
Type=notify
PIDFile=/var/run/php${php_version}-fpm.pid
ExecStart=/usr/local/php${php_version}/sbin/php-fpm --nodaemonize --fpm-config /usr/local/php${php_version}/etc/php-fpm.conf
ExecReload=/bin/kill -USR2 \$MAINPID
PrivateTmp=true
RuntimeDirectory=php${php_version}-fpm
RuntimeDirectoryMode=0755

[Install]
WantedBy=multi-user.target
EOF

    # Create generic php-fpm service for default version
    if [[ "$php_version" == "$(echo $PHP_DEFAULT_VERSION | cut -d. -f1-2)" ]]; then
        cat > /etc/systemd/system/php-fpm.service << EOF
[Unit]
Description=The PHP FastCGI Process Manager (default version)
After=network.target

[Service]
Type=notify
PIDFile=/var/run/php${php_version}-fpm.pid
ExecStart=/usr/local/php${php_version}/sbin/php-fpm --nodaemonize --fpm-config /usr/local/php${php_version}/etc/php-fpm.conf
ExecReload=/bin/kill -USR2 \$MAINPID
PrivateTmp=true
RuntimeDirectory=php-fpm
RuntimeDirectoryMode=0755

[Install]
WantedBy=multi-user.target
EOF
    fi

    systemctl daemon-reload
    systemctl enable "php${php_version}-fpm"
    
    log_message "SUCCESS" "PHP-FPM $php_version systemd service created"
}

# Start PHP-FPM service
start_php_fpm() {
    local php_version=${1:-$(echo $PHP_DEFAULT_VERSION | cut -d. -f1-2)}
    
    log_message "INFO" "Starting PHP-FPM $php_version service..."
    
    if systemctl start "php${php_version}-fpm"; then
        log_message "SUCCESS" "PHP-FPM $php_version started successfully"
        return 0
    else
        log_message "ERROR" "Failed to start PHP-FPM $php_version"
        return 1
    fi
}

# Stop PHP-FPM service
stop_php_fpm() {
    local php_version=${1:-$(echo $PHP_DEFAULT_VERSION | cut -d. -f1-2)}
    
    log_message "INFO" "Stopping PHP-FPM $php_version service..."
    
    if systemctl stop "php${php_version}-fpm"; then
        log_message "SUCCESS" "PHP-FPM $php_version stopped successfully"
        return 0
    else
        log_message "ERROR" "Failed to stop PHP-FPM $php_version"
        return 1
    fi
}

# Upgrade/downgrade PHP-FPM
upgrade_php_fpm() {
    local current_version=$1
    local new_version=$2
    
    if [[ -z "$current_version" ]] || [[ -z "$new_version" ]]; then
        log_message "ERROR" "Please specify both current and new PHP versions"
        return 1
    fi
    
    local current_major_minor=$(echo "$current_version" | cut -d. -f1-2)
    
    log_message "INFO" "Upgrading PHP-FPM from $current_version to $new_version..."
    
    # Stop current PHP-FPM
    stop_php_fpm "$current_major_minor"
    
    # Install new version
    if install_php_fpm "$new_version"; then
        log_message "SUCCESS" "PHP-FPM upgraded to version $new_version successfully"
        return 0
    else
        log_message "ERROR" "Failed to upgrade PHP-FPM"
        return 1
    fi
}

# Remove PHP-FPM version
remove_php_fpm() {
    local php_version=$1
    
    if [[ -z "$php_version" ]]; then
        log_message "ERROR" "Please specify PHP version to remove"
        return 1
    fi
    
    local php_major_minor=$(echo "$php_version" | cut -d. -f1-2)
    
    log_message "INFO" "Removing PHP-FPM $php_version..."
    
    # Stop service
    stop_php_fpm "$php_major_minor"
    
    # Disable and remove service
    systemctl disable "php${php_major_minor}-fpm" 2>/dev/null
    rm -f "/etc/systemd/system/php${php_major_minor}-fpm.service"
    
    # Remove installation directory
    rm -rf "/usr/local/php${php_major_minor}"
    
    # Remove logs and session data
    rm -rf "$LOG_DIR/php${php_major_minor}"
    rm -rf "/var/lib/php${php_major_minor}"
    
    systemctl daemon-reload
    
    log_message "SUCCESS" "PHP-FPM $php_version removed successfully"
}

# List installed PHP versions
list_php_versions() {
    echo "=== Installed PHP Versions ==="
    
    for php_dir in /usr/local/php*/; do
        if [[ -d "$php_dir" ]]; then
            local version_path="${php_dir%/}"
            local version_num="${version_path##*/php}"
            local php_binary="$php_dir/bin/php"
            
            if [[ -f "$php_binary" ]]; then
                local full_version=$($php_binary -v | head -1 | cut -d' ' -f2)
                local status=$(systemctl is-active "php${version_num}-fpm" 2>/dev/null || echo "inactive")
                echo "PHP $version_num: $full_version (Status: $status)"
            fi
        fi
    done
}

# Get PHP-FPM status
php_fpm_status() {
    local php_version=${1:-$(echo $PHP_DEFAULT_VERSION | cut -d. -f1-2)}
    
    echo "=== PHP-FPM $php_version Status ==="
    
    local php_binary="/usr/local/php${php_version}/bin/php"
    if [[ -f "$php_binary" ]]; then
        echo "Version: $($php_binary -v | head -1 | cut -d' ' -f2)"
        echo "Status: $(systemctl is-active php${php_version}-fpm 2>/dev/null || echo 'Not installed')"
        echo "Configuration: /usr/local/php${php_version}/etc/php-fpm.conf"
        echo "Pool config: /usr/local/php${php_version}/etc/php-fpm.d/www.conf"
        echo "Socket: $PHP_FPM_SOCKET_DIR/php${php_version}-fpm.sock"
        echo "Process ID: $(cat /var/run/php${php_version}-fpm.pid 2>/dev/null || echo 'Not running')"
        echo "Active pools: $($php_binary -i | grep -c 'pool.*www' 2>/dev/null || echo '0')"
    else
        echo "PHP $php_version is not installed"
    fi
}

# Test PHP-FPM configuration
test_php_fpm_config() {
    local php_version=${1:-$(echo $PHP_DEFAULT_VERSION | cut -d. -f1-2)}
    
    log_message "INFO" "Testing PHP-FPM $php_version configuration..."
    
    local fpm_binary="/usr/local/php${php_version}/sbin/php-fpm"
    if [[ -f "$fmp_binary" ]]; then
        if $fpm_binary -t; then
            log_message "SUCCESS" "PHP-FPM $php_version configuration test passed"
            return 0
        else
            log_message "ERROR" "PHP-FPM $php_version configuration test failed"
            return 1
        fi
    else
        log_message "ERROR" "PHP-FPM $php_version is not installed"
        return 1
    fi
}

# Main execution
case "${1:-help}" in
    "install")
        install_php_fpm "$2"
        ;;
    "start")
        start_php_fpm "$2"
        ;;
    "stop")
        stop_php_fpm "$2"
        ;;
    "restart")
        stop_php_fpm "$2" && start_php_fpm "$2"
        ;;
    "upgrade")
        upgrade_php_fpm "$2" "$3"
        ;;
    "remove")
        remove_php_fpm "$2"
        ;;
    "list")
        list_php_versions
        ;;
    "status")
        php_fpm_status "$2"
        ;;
    "test")
        test_php_fpm_config "$2"
        ;;
    *)
        echo "ThangLQ Stack - PHP-FPM Module"
        echo "Usage: $0 <command> [options]"
        echo ""
        echo "Commands:"
        echo "  install <version>           - Install PHP-FPM version"
        echo "  start [version]             - Start PHP-FPM service"
        echo "  stop [version]              - Stop PHP-FPM service"
        echo "  restart [version]           - Restart PHP-FPM service"
        echo "  upgrade <old> <new>         - Upgrade PHP-FPM version"
        echo "  remove <version>            - Remove PHP-FPM version"
        echo "  list                        - List installed PHP versions"
        echo "  status [version]            - Show PHP-FPM status"
        echo "  test [version]              - Test PHP-FPM configuration"
        echo ""
        echo "Examples:"
        echo "  $0 install 8.3.13"
        echo "  $0 start 8.3"
        echo "  $0 upgrade 8.2.13 8.3.13"
        echo "  $0 list"
        echo "  $0 status 8.3"
        ;;
esac 