#!/bin/bash

# ================================================================
# ThangLQ Stack - PHP Extension Manager
# Manages PHP extensions installation and removal
# ================================================================

# Get script directory and load common utilities
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
source "$SCRIPT_DIR/../utils/common.sh"

# Install PECL extension
install_pecl_extension() {
    local extension=$1
    local php_version=${2:-"8.3"}
    
    if [[ -z "$extension" ]]; then
        log_message "ERROR" "Extension name required"
        return 1
    fi
    
    log_message "INFO" "Installing PHP extension: $extension for PHP $php_version"
    
    # Check if PHP version exists
    local php_path="/usr/local/php${php_version}"
    if [[ ! -d "$php_path" ]]; then
        log_message "ERROR" "PHP $php_version not found. Please install it first."
        return 1
    fi
    
    # Set PATH for PHP tools
    export PATH="$php_path/bin:$PATH"
    
    # Install development packages if needed
    install_dev_packages
    
    # Install extension based on type
    case $extension in
        "redis")
            install_redis_extension "$php_version"
            ;;
        "memcached")
            install_memcached_extension "$php_version"
            ;;
        "igbinary")
            install_igbinary_extension "$php_version"
            ;;
        "apcu")
            install_apcu_extension "$php_version"
            ;;
        "xdebug")
            install_xdebug_extension "$php_version"
            ;;
        "mongodb")
            install_mongodb_extension "$php_version"
            ;;
        "imagick")
            install_imagick_extension "$php_version"
            ;;
        *)
            # Try generic PECL installation
            install_generic_pecl_extension "$extension" "$php_version"
            ;;
    esac
    
    if [[ $? -eq 0 ]]; then
        # Restart PHP-FPM to load new extension
        restart_php_fpm "$php_version"
        log_message "SUCCESS" "Extension $extension installed successfully for PHP $php_version"
        return 0
    else
        log_message "ERROR" "Failed to install extension $extension"
        return 1
    fi
}

# Install development packages
install_dev_packages() {
    if command -v dnf >/dev/null 2>&1; then
        dnf install -y autoconf gcc gcc-c++ libtool make pkg-config \
            zlib-devel libmemcached-devel cyrus-sasl-devel \
            ImageMagick-devel pcre-devel
    else
        yum install -y autoconf gcc gcc-c++ libtool make pkg-config \
            zlib-devel libmemcached-devel cyrus-sasl-devel \
            ImageMagick-devel pcre-devel
    fi
}

# Install Redis extension
install_redis_extension() {
    local php_version=$1
    local php_path="/usr/local/php${php_version}"
    
    cd /tmp
    wget https://github.com/phpredis/phpredis/archive/develop.zip -O phpredis.zip
    unzip -o phpredis.zip
    cd phpredis-develop
    
    "$php_path/bin/phpize"
    ./configure --with-php-config="$php_path/bin/php-config" --enable-redis-igbinary
    make -j$(nproc) && make install
    
    # Add extension to PHP configuration
    echo "extension=redis.so" > "$php_path/etc/conf.d/20-redis.ini"
    
    return $?
}

# Install Memcached extension
install_memcached_extension() {
    local php_version=$1
    local php_path="/usr/local/php${php_version}"
    
    cd /tmp
    wget https://github.com/php-memcached-dev/php-memcached/archive/master.zip -O memcached.zip
    unzip -o memcached.zip
    cd php-memcached-master
    
    "$php_path/bin/phpize"
    ./configure --with-php-config="$php_path/bin/php-config" --enable-memcached-igbinary
    make -j$(nproc) && make install
    
    # Add extension to PHP configuration
    echo "extension=memcached.so" > "$php_path/etc/conf.d/20-memcached.ini"
    
    return $?
}

# Install Igbinary extension
install_igbinary_extension() {
    local php_version=$1
    local php_path="/usr/local/php${php_version}"
    
    cd /tmp
    wget https://github.com/igbinary/igbinary/archive/master.zip -O igbinary.zip
    unzip -o igbinary.zip
    cd igbinary-master
    
    "$php_path/bin/phpize"
    ./configure --with-php-config="$php_path/bin/php-config"
    make -j$(nproc) && make install
    
    # Add extension to PHP configuration
    echo "extension=igbinary.so" > "$php_path/etc/conf.d/10-igbinary.ini"
    
    return $?
}

# Install APCu extension
install_apcu_extension() {
    local php_version=$1
    local php_path="/usr/local/php${php_version}"
    
    cd /tmp
    wget https://github.com/krakjoe/apcu/archive/master.zip -O apcu.zip
    unzip -o apcu.zip
    cd apcu-master
    
    "$php_path/bin/phpize"
    ./configure --with-php-config="$php_path/bin/php-config"
    make -j$(nproc) && make install
    
    # Add extension to PHP configuration
    cat > "$php_path/etc/conf.d/20-apcu.ini" << EOF
extension=apcu.so
apc.enabled=1
apc.shm_size=128M
apc.ttl=3600
apc.user_ttl=3600
apc.gc_ttl=3600
apc.entries_hint=4096
apc.slam_defense=1
EOF
    
    return $?
}

# Install Xdebug extension
install_xdebug_extension() {
    local php_version=$1
    local php_path="/usr/local/php${php_version}"
    
    cd /tmp
    wget https://github.com/xdebug/xdebug/archive/master.zip -O xdebug.zip
    unzip -o xdebug.zip
    cd xdebug-master
    
    "$php_path/bin/phpize"
    ./configure --with-php-config="$php_path/bin/php-config"
    make -j$(nproc) && make install
    
    # Add extension to PHP configuration
    cat > "$php_path/etc/conf.d/20-xdebug.ini" << EOF
zend_extension=xdebug.so
xdebug.mode=debug
xdebug.start_with_request=yes
xdebug.client_host=$XDEBUG_REMOTE_HOST
xdebug.client_port=$XDEBUG_REMOTE_PORT
xdebug.log=/var/log/thanglq/php${php_version}/xdebug.log
EOF
    
    return $?
}

# Install MongoDB extension
install_mongodb_extension() {
    local php_version=$1
    local php_path="/usr/local/php${php_version}"
    
    cd /tmp
    wget https://github.com/mongodb/mongo-php-driver/archive/master.zip -O mongodb.zip
    unzip -o mongodb.zip
    cd mongo-php-driver-master
    
    "$php_path/bin/phpize"
    ./configure --with-php-config="$php_path/bin/php-config"
    make -j$(nproc) && make install
    
    # Add extension to PHP configuration
    echo "extension=mongodb.so" > "$php_path/etc/conf.d/20-mongodb.ini"
    
    return $?
}

# Install ImageMagick extension
install_imagick_extension() {
    local php_version=$1
    local php_path="/usr/local/php${php_version}"
    
    cd /tmp
    wget https://github.com/Imagick/imagick/archive/master.zip -O imagick.zip
    unzip -o imagick.zip
    cd imagick-master
    
    "$php_path/bin/phpize"
    ./configure --with-php-config="$php_path/bin/php-config"
    make -j$(nproc) && make install
    
    # Add extension to PHP configuration
    echo "extension=imagick.so" > "$php_path/etc/conf.d/20-imagick.ini"
    
    return $?
}

# Install generic PECL extension
install_generic_pecl_extension() {
    local extension=$1
    local php_version=$2
    local php_path="/usr/local/php${php_version}"
    
    log_message "INFO" "Attempting generic PECL installation for: $extension"
    
    # Try PECL first
    if "$php_path/bin/pecl" install "$extension"; then
        echo "extension=$extension.so" > "$php_path/etc/conf.d/20-$extension.ini"
        return 0
    else
        log_message "ERROR" "PECL installation failed for $extension"
        return 1
    fi
}

# Remove extension
remove_extension() {
    local extension=$1
    local php_version=${2:-"8.3"}
    
    if [[ -z "$extension" ]]; then
        log_message "ERROR" "Extension name required"
        return 1
    fi
    
    log_message "INFO" "Removing PHP extension: $extension from PHP $php_version"
    
    local php_path="/usr/local/php${php_version}"
    if [[ ! -d "$php_path" ]]; then
        log_message "ERROR" "PHP $php_version not found"
        return 1
    fi
    
    # Remove extension configuration
    rm -f "$php_path/etc/conf.d/"*"$extension.ini"
    
    # Remove extension file (if we can identify it)
    local ext_dir=$("$php_path/bin/php-config" --extension-dir)
    rm -f "$ext_dir/$extension.so"
    
    # Restart PHP-FPM
    restart_php_fpm "$php_version"
    
    log_message "SUCCESS" "Extension $extension removed from PHP $php_version"
    return 0
}

# List installed extensions
list_extensions() {
    local php_version=${1:-"8.3"}
    
    log_message "INFO" "Listing extensions for PHP $php_version"
    
    local php_path="/usr/local/php${php_version}"
    if [[ ! -d "$php_path" ]]; then
        log_message "ERROR" "PHP $php_version not found"
        return 1
    fi
    
    echo "=== PHP $php_version Extensions ==="
    echo ""
    
    # Core extensions
    echo "Core Extensions:"
    "$php_path/bin/php" -m | grep -v "Zend" | head -20
    
    echo ""
    echo "Zend Extensions:"
    "$php_path/bin/php" -m | grep "Zend"
    
    echo ""
    echo "Extension Configuration Files:"
    ls -la "$php_path/etc/conf.d/"*.ini 2>/dev/null || echo "No extension config files found"
    
    return 0
}

# Install common bundle
install_common_bundle() {
    local php_version=${1:-"8.3"}
    
    log_message "INFO" "Installing common PHP extensions bundle for PHP $php_version"
    
    local extensions=("igbinary" "redis" "memcached" "apcu")
    
    for ext in "${extensions[@]}"; do
        log_message "INFO" "Installing $ext..."
        install_pecl_extension "$ext" "$php_version"
        sleep 2
    done
    
    log_message "SUCCESS" "Common extensions bundle installed for PHP $php_version"
}

# Restart PHP-FPM
restart_php_fpm() {
    local php_version=$1
    
    if systemctl is-active "php${php_version}-fpm" >/dev/null 2>&1; then
        systemctl restart "php${php_version}-fpm"
        log_message "INFO" "PHP-FPM $php_version restarted"
    else
        log_message "WARNING" "PHP-FPM $php_version service not running"
    fi
}

# Main execution
case "${1:-help}" in
    "install")
        install_pecl_extension "$2" "$3"
        ;;
    "remove")
        remove_extension "$2" "$3"
        ;;
    "list")
        list_extensions "$2"
        ;;
    "bundle")
        install_common_bundle "$2"
        ;;
    *)
        echo "ThangLQ Stack - PHP Extension Manager"
        echo "Usage: $0 <command> [extension] [php_version]"
        echo ""
        echo "Commands:"
        echo "  install <extension> [php_version] - Install PHP extension"
        echo "  remove <extension> [php_version]  - Remove PHP extension"
        echo "  list [php_version]                - List installed extensions"
        echo "  bundle [php_version]              - Install common extensions bundle"
        echo ""
        echo "Supported Extensions:"
        echo "  redis, memcached, igbinary, apcu, xdebug, mongodb, imagick"
        echo "  + any PECL extension"
        echo ""
        echo "Examples:"
        echo "  $0 install redis 8.3"
        echo "  $0 remove xdebug 8.3"
        echo "  $0 list 8.3"
        echo "  $0 bundle 8.3"
        ;;
esac 