#!/bin/bash

# ================================================================
# ThangLQ Stack - Virtual Host Manager
# Automated Nginx vhost creation with SSL support
# ================================================================

# Get script directory and load common utilities
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
source "$SCRIPT_DIR/../utils/common.sh"

# Create virtual host function
create_vhost() {
    local domain=$1
    local ssl_enabled=${2:-false}
    local php_version=${3:-"8.3"}
    
    if [[ -z "$domain" ]]; then
        echo "Usage: create_vhost <domain> [ssl_enabled] [php_version]"
        return 1
    fi
    
    log_message "INFO" "Creating virtual host for $domain"
    
    # Create domain directory structure
    local vhost_dir="$VHOSTS_DIR/$domain"
    mkdir -p "$vhost_dir"/{public,logs,ssl,backup}
    
    # Set permissions
    chown -R "$NGINX_USER:$NGINX_GROUP" "$vhost_dir"
    chmod -R 755 "$vhost_dir"
    
    # Create Nginx configuration
    create_nginx_vhost_config "$domain" "$ssl_enabled" "$php_version"
    
    # Create SSL certificate if enabled
    if [[ "$ssl_enabled" == "true" ]]; then
        create_ssl_certificate "$domain"
    fi
    
    # Create sample files
    create_sample_files "$domain" "$php_version"
    
    # Test and reload Nginx
    if test_config "nginx"; then
        restart_service "nginx"
        log_message "SUCCESS" "Virtual host $domain created successfully"
        
        echo ""
        echo "╔════════════════════════════════════════════════════════╗"
        echo "║               Virtual Host Created!                   ║"
        echo "╠════════════════════════════════════════════════════════╣"
        echo "║ Domain:       $domain                         ║"
        echo "║ Document Root: $vhost_dir/public                    ║"
        echo "║ Logs:         $vhost_dir/logs/                      ║"
        if [[ "$ssl_enabled" == "true" ]]; then
            echo "║ SSL:          Enabled (Self-signed)                   ║"
            echo "║ HTTPS URL:    https://$domain                  ║"
        fi
        echo "║ HTTP URL:     http://$domain                   ║"
        echo "║ PHP Version:  $php_version                            ║"
        echo "╚════════════════════════════════════════════════════════╝"
    else
        log_message "ERROR" "Nginx configuration test failed"
        return 1
    fi
}

# Create Nginx vhost configuration
create_nginx_vhost_config() {
    local domain=$1
    local ssl_enabled=$2
    local php_version=$3
    
    # Determine nginx config directory
    local config_dir
    if [[ -d "/usr/local/openresty/nginx/conf" ]]; then
        config_dir="/usr/local/openresty/nginx/conf"
    else
        config_dir="/usr/local/nginx/conf"
    fi
    
    local config_file="$config_dir/conf.d/${domain}.conf"
    
    cat > "$config_file" << EOF
# Virtual Host Configuration for $domain
# Generated by $BRAND_NAME
# Date: $(date)

server {
    listen 80;
    server_name $domain www.$domain;
    root $VHOSTS_DIR/$domain/public;
    index index.php index.html index.htm;
    
    # Logging
    access_log $VHOSTS_DIR/$domain/logs/access.log main;
    error_log $VHOSTS_DIR/$domain/logs/error.log warn;
    
EOF

    if [[ "$ssl_enabled" == "true" ]]; then
        cat >> "$config_file" << EOF
    # Redirect HTTP to HTTPS
    return 301 https://\$server_name\$request_uri;
}

server {
    listen 443 ssl http2;
    server_name $domain www.$domain;
    root $VHOSTS_DIR/$domain/public;
    index index.php index.html index.htm;
    
    # SSL Configuration
    ssl_certificate $VHOSTS_DIR/$domain/ssl/$domain.crt;
    ssl_certificate_key $VHOSTS_DIR/$domain/ssl/$domain.key;
    ssl_protocols $SSL_PROTOCOLS;
    ssl_ciphers $SSL_CIPHERS;
    ssl_prefer_server_ciphers $SSL_PREFER_SERVER_CIPHERS;
    ssl_session_cache shared:SSL:10m;
    ssl_session_timeout 10m;
    
    # HSTS
    add_header Strict-Transport-Security "max-age=$HSTS_MAX_AGE; includeSubDomains; preload" always;
    
    # Logging
    access_log $VHOSTS_DIR/$domain/logs/ssl_access.log main;
    error_log $VHOSTS_DIR/$domain/logs/ssl_error.log warn;
    
EOF
    fi

    cat >> "$config_file" << EOF
    # Security headers
    add_header X-Frame-Options DENY always;
    add_header X-Content-Type-Options nosniff always;
    add_header X-XSS-Protection "1; mode=block" always;
    add_header Referrer-Policy "strict-origin-when-cross-origin" always;
    
    # Rate limiting
    limit_req zone=general burst=5 nodelay;
    
    # Static files optimization
    location ~* \.(jpg|jpeg|png|gif|ico|css|js|pdf|txt)$ {
        expires 1y;
        add_header Cache-Control "public, no-transform";
        access_log off;
    }
    
    # PHP-FPM processing
    location ~ \.php$ {
        try_files \$uri =404;
        fastcgi_split_path_info ^(.+\.php)(/.+)$;
        fastcgi_pass unix:$PHP_FPM_SOCKET_DIR/php${php_version}-fpm.sock;
        fastcgi_index index.php;
        fastcgi_param SCRIPT_FILENAME \$document_root\$fastcgi_script_name;
        include fastcgi_params;
        
        # Security
        fastcgi_param PHP_ADMIN_VALUE "open_basedir=$VHOSTS_DIR/$domain/:/tmp/";
    }
    
    # Deny access to sensitive files
    location ~ /\. {
        deny all;
        access_log off;
        log_not_found off;
    }
    
    location ~ ~$ {
        deny all;
        access_log off;
        log_not_found off;
    }
    
    # WordPress specific optimizations (if WordPress)
    location = /favicon.ico {
        log_not_found off;
        access_log off;
    }
    
    location = /robots.txt {
        allow all;
        log_not_found off;
        access_log off;
    }
    
    # Default location
    location / {
        try_files \$uri \$uri/ /index.php?\$args;
    }
}
EOF

    log_message "SUCCESS" "Nginx configuration created for $domain"
}

# Create SSL certificate
create_ssl_certificate() {
    local domain=$1
    local ssl_dir="$VHOSTS_DIR/$domain/ssl"
    
    log_message "INFO" "Creating self-signed SSL certificate for $domain"
    
    # Generate private key
    openssl genrsa -out "$ssl_dir/$domain.key" 2048
    
    # Generate certificate signing request
    openssl req -new -key "$ssl_dir/$domain.key" -out "$ssl_dir/$domain.csr" \
        -subj "/C=VN/ST=Ho Chi Minh/L=Ho Chi Minh City/O=$BRAND_NAME/CN=$domain"
    
    # Generate self-signed certificate
    openssl x509 -req -days 365 -in "$ssl_dir/$domain.csr" \
        -signkey "$ssl_dir/$domain.key" -out "$ssl_dir/$domain.crt"
    
    # Set permissions
    chmod 600 "$ssl_dir/$domain.key"
    chmod 644 "$ssl_dir/$domain.crt"
    
    log_message "SUCCESS" "SSL certificate created for $domain"
}

# Create sample files
create_sample_files() {
    local domain=$1
    local php_version=$2
    local doc_root="$VHOSTS_DIR/$domain/public"
    
    # Create sample index.php
    cat > "$doc_root/index.php" << EOF
<?php
/**
 * $BRAND_NAME - Sample Page
 * Domain: $domain
 * PHP Version: $php_version
 */

// System information
\$system_info = [
    'server_software' => \$_SERVER['SERVER_SOFTWARE'] ?? 'Unknown',
    'php_version' => PHP_VERSION,
    'server_time' => date('Y-m-d H:i:s'),
    'document_root' => \$_SERVER['DOCUMENT_ROOT'],
    'remote_addr' => \$_SERVER['REMOTE_ADDR'] ?? 'Unknown',
    'user_agent' => \$_SERVER['HTTP_USER_AGENT'] ?? 'Unknown'
];

// Check database connection
\$db_status = 'Not tested';
try {
    \$pdo = new PDO('mysql:host=localhost', 'root', '');
    \$db_status = '✓ Connected';
} catch (PDOException \$e) {
    \$db_status = '✗ Connection failed';
}

// Check memcached
\$memcached_status = '✗ Not available';
if (class_exists('Memcached')) {
    \$memcached_status = '✓ Available';
}

// Check Redis
\$redis_status = '✗ Not available';
if (class_exists('Redis')) {
    \$redis_status = '✓ Available';
}
?>
<!DOCTYPE html>
<html lang="vi">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>$BRAND_NAME - <?php echo \$domain; ?></title>
    <style>
        body { font-family: Arial, sans-serif; margin: 0; padding: 20px; background: #f5f5f5; }
        .container { max-width: 800px; margin: 0 auto; background: white; padding: 30px; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        .header { text-align: center; margin-bottom: 30px; }
        .logo { font-size: 2.5em; color: #007bff; font-weight: bold; }
        .subtitle { color: #666; margin-top: 10px; }
        .info-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 20px; margin: 20px 0; }
        .info-card { background: #f8f9fa; padding: 20px; border-radius: 8px; border-left: 4px solid #007bff; }
        .info-card h3 { margin: 0 0 10px 0; color: #333; }
        .status { font-weight: bold; }
        .success { color: #28a745; }
        .error { color: #dc3545; }
        .footer { text-align: center; margin-top: 30px; color: #666; border-top: 1px solid #eee; padding-top: 20px; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <div class="logo">🚀 $BRAND_NAME</div>
            <div class="subtitle">High-Performance Web Stack</div>
            <p>Domain: <strong><?php echo \$domain; ?></strong></p>
        </div>
        
        <div class="info-grid">
            <div class="info-card">
                <h3>🖥️ Server Information</h3>
                <p><strong>Server:</strong> <?php echo \$system_info['server_software']; ?></p>
                <p><strong>PHP Version:</strong> <?php echo \$system_info['php_version']; ?></p>
                <p><strong>Time:</strong> <?php echo \$system_info['server_time']; ?></p>
            </div>
            
            <div class="info-card">
                <h3>🔧 Services Status</h3>
                <p><strong>Database:</strong> <span class="status <?php echo strpos(\$db_status, '✓') !== false ? 'success' : 'error'; ?>"><?php echo \$db_status; ?></span></p>
                <p><strong>Memcached:</strong> <span class="status <?php echo strpos(\$memcached_status, '✓') !== false ? 'success' : 'error'; ?>"><?php echo \$memcached_status; ?></span></p>
                <p><strong>Redis:</strong> <span class="status <?php echo strpos(\$redis_status, '✓') !== false ? 'success' : 'error'; ?>"><?php echo \$redis_status; ?></span></p>
            </div>
            
            <div class="info-card">
                <h3>🌐 Request Information</h3>
                <p><strong>Your IP:</strong> <?php echo \$system_info['remote_addr']; ?></p>
                <p><strong>Document Root:</strong> <?php echo \$system_info['document_root']; ?></p>
            </div>
        </div>
        
        <div class="footer">
            <p>Powered by <strong>$BRAND_NAME</strong> | OpenResty + PHP-FPM + MariaDB + Memcached + Redis</p>
        </div>
    </div>
</body>
</html>
EOF

    # Create phpinfo page
    cat > "$doc_root/phpinfo.php" << EOF
<?php
// Remove this file in production
if (\$_GET['show'] === 'info') {
    phpinfo();
} else {
    echo '<h1>Access Denied</h1><p>Add ?show=info to view phpinfo</p>';
}
?>
EOF

    chown -R "$NGINX_USER:$NGINX_GROUP" "$doc_root"
    log_message "SUCCESS" "Sample files created for $domain"
}

# Delete virtual host
delete_vhost() {
    local domain=$1
    
    if [[ -z "$domain" ]]; then
        echo "Usage: delete_vhost <domain>"
        return 1
    fi
    
    echo "WARNING: This will permanently delete the virtual host for $domain"
    read -p "Are you sure? (yes/no): " confirm
    
    if [[ "$confirm" == "yes" ]]; then
        # Determine nginx config directory
        local config_dir
        if [[ -d "/usr/local/openresty/nginx/conf" ]]; then
            config_dir="/usr/local/openresty/nginx/conf"
        else
            config_dir="/usr/local/nginx/conf"
        fi
        
        rm -f "$config_dir/conf.d/${domain}.conf"
        rm -rf "$VHOSTS_DIR/$domain"
        
        if test_config "nginx"; then
            restart_service "nginx"
            log_message "SUCCESS" "Virtual host $domain deleted successfully"
        else
            log_message "ERROR" "Nginx configuration test failed after deletion"
        fi
    else
        echo "Operation cancelled"
    fi
}

# List virtual hosts
list_vhosts() {
    echo "Active Virtual Hosts:"
    echo "===================="
    
    # Determine nginx config directory
    local config_dir
    if [[ -d "/usr/local/openresty/nginx/conf" ]]; then
        config_dir="/usr/local/openresty/nginx/conf"
    else
        config_dir="/usr/local/nginx/conf"
    fi
    
    for config in "$config_dir/conf.d"/*.conf; do
        if [[ -f "$config" ]]; then
            local domain=$(basename "$config" .conf)
            local ssl_status="HTTP Only"
            
            if grep -q "listen 443" "$config"; then
                ssl_status="HTTP + HTTPS"
            fi
            
            echo "Domain: $domain | SSL: $ssl_status"
        fi
    done
}

# Main execution
case "${1:-menu}" in
    "create")
        create_vhost "$2" "$3" "$4"
        ;;
    "delete")
        delete_vhost "$2"
        ;;
    "list")
        list_vhosts
        ;;
    *)
        echo "$BRAND_NAME - Virtual Host Manager"
        echo "Usage: $0 <command> [options]"
        echo ""
        echo "Commands:"
        echo "  create <domain> [ssl_enabled] [php_version] - Create new virtual host"
        echo "  delete <domain>                            - Delete virtual host"
        echo "  list                                       - List all virtual hosts"
        echo ""
        echo "Examples:"
        echo "  $0 create example.com true 8.3"
        echo "  $0 create test.local false 8.1"
        echo "  $0 delete example.com"
        echo "  $0 list"
        ;;
esac 