#!/bin/bash

# ================================================================
# Test Module Download Functionality
# ================================================================

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

echo "🔧 Testing Module Download Functionality"
echo "========================================"
echo ""

# Source the install.sh functions
source ./install.sh

# Test configuration
export SCRIPTS_BASE_URL="http://scripts.thanglq.dev"
export SCRIPTS_CACHE_DIR="/tmp/test-thanglq-scripts"
export AUTO_UPDATE_SCRIPTS="true"

log_test() {
    local level=$1
    local message=$2
    case $level in
        "PASS") echo -e "${GREEN}✅ PASS${NC} $message" ;;
        "FAIL") echo -e "${RED}❌ FAIL${NC} $message" ;;
        "INFO") echo -e "${BLUE}ℹ️  INFO${NC} $message" ;;
        "WARN") echo -e "${YELLOW}⚠️  WARN${NC} $message" ;;
    esac
}

# Test 1: Test download_script function
echo "Test 1: Testing download_script function..."
test_script_path="modules/cache/redis.sh"
echo "Attempting to download: $test_script_path"

if result=$(download_script "$test_script_path" 2>&1); then
    if [[ -f "$result" ]]; then
        log_test "PASS" "Script downloaded successfully to: $result"
        # Check if file is executable
        if [[ -x "$result" ]]; then
            log_test "PASS" "Downloaded script is executable"
        else
            log_test "FAIL" "Downloaded script is not executable"
        fi
    else
        log_test "FAIL" "download_script returned success but file not found"
    fi
else
    log_test "FAIL" "download_script function failed: $result"
fi

echo ""

# Test 2: Test load_remote_module function
echo "Test 2: Testing load_remote_module function..."
echo "Attempting to load: $test_script_path"

if load_remote_module "$test_script_path" 2>&1; then
    log_test "PASS" "Module loaded successfully"
    
    # Check if functions from the module are available
    if declare -f install_redis >/dev/null 2>&1; then
        log_test "PASS" "Module functions are available (install_redis found)"
    else
        log_test "WARN" "Module loaded but functions not found (expected if Redis module doesn't exist)"
    fi
else
    log_test "FAIL" "load_remote_module function failed"
fi

echo ""

# Test 3: Test check_script_availability function
echo "Test 3: Testing check_script_availability function..."
if check_script_availability "$test_script_path"; then
    log_test "PASS" "Script availability check passed"
else
    log_test "FAIL" "Script availability check failed (expected if server not accessible)"
fi

echo ""

# Test 4: Test cache directory structure
echo "Test 4: Testing cache directory structure..."
if [[ -d "$SCRIPTS_CACHE_DIR" ]]; then
    log_test "PASS" "Cache directory exists: $SCRIPTS_CACHE_DIR"
    
    # List cached files
    echo "Cached files:"
    find "$SCRIPTS_CACHE_DIR" -name "*.sh" -type f 2>/dev/null | while read -r file; do
        echo "  - $file"
    done
    
    cache_size=$(du -sh "$SCRIPTS_CACHE_DIR" 2>/dev/null | cut -f1 || echo '0B')
    log_test "INFO" "Cache size: $cache_size"
else
    log_test "FAIL" "Cache directory not created"
fi

echo ""

# Test 5: Test error handling
echo "Test 5: Testing error handling with non-existent module..."
fake_module="modules/nonexistent/fake.sh"
if load_remote_module "$fake_module" 2>&1; then
    log_test "FAIL" "load_remote_module should have failed for non-existent module"
else
    log_test "PASS" "Error handling works correctly for non-existent modules"
fi

echo ""

# Test 6: Test clear_scripts_cache function
echo "Test 6: Testing clear_scripts_cache function..."
if clear_scripts_cache 2>&1; then
    if [[ ! -d "$SCRIPTS_CACHE_DIR" ]] || [[ -z "$(ls -A "$SCRIPTS_CACHE_DIR" 2>/dev/null)" ]]; then
        log_test "PASS" "Scripts cache cleared successfully"
    else
        log_test "FAIL" "Scripts cache not properly cleared"
    fi
else
    log_test "FAIL" "clear_scripts_cache function failed"
fi

echo ""

# Summary
echo "🎯 Test Summary"
echo "==============="
log_test "INFO" "Module download functionality tested"
log_test "INFO" "Check output above for any failures"

echo ""
echo "📋 Common Issues & Solutions:"
echo "=============================="
echo "• [ERROR] Failed to load remote module:"
echo "  → Check internet connectivity"
echo "  → Verify SCRIPTS_BASE_URL is accessible"
echo "  → Ensure curl is installed"
echo ""
echo "• [ERROR] Download failed:"
echo "  → Check if remote server is running"
echo "  → Verify script paths are correct"
echo "  → Check firewall/proxy settings"

# Cleanup
rm -rf "$SCRIPTS_CACHE_DIR"
log_test "INFO" "Test cleanup completed" 