#!/bin/bash

# ================================================================
# Test Fallback Mechanism
# ================================================================

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

echo "🔄 Testing Fallback Mechanism"
echo "==============================="
echo ""

log_test() {
    local level=$1
    local message=$2
    case $level in
        "PASS") echo -e "${GREEN}✅ PASS${NC} $message" ;;
        "FAIL") echo -e "${RED}❌ FAIL${NC} $message" ;;
        "INFO") echo -e "${BLUE}ℹ️  INFO${NC} $message" ;;
        "WARN") echo -e "${YELLOW}⚠️  WARN${NC} $message" ;;
    esac
}

# Load the install.sh functions without running the full script
echo "Loading install.sh functions..."
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"

# Set test configuration to use non-existent server
export SCRIPTS_BASE_URL="http://nonexistent.example.com"
export SCRIPTS_CACHE_DIR="/tmp/test-fallback-cache"
export AUTO_UPDATE_SCRIPTS="true"

# Source only the function definitions, not the main execution
if source <(sed '/^main "\$@"/,$d' install.sh) 2>/dev/null; then
    log_test "PASS" "Install.sh functions loaded"
else
    log_test "FAIL" "Failed to load install.sh functions"
    exit 1
fi

echo ""

# Test 1: Verify functions are available
echo "Test 1: Checking function availability..."
if declare -f load_remote_module >/dev/null 2>&1; then
    log_test "PASS" "load_remote_module function available"
else
    log_test "FAIL" "load_remote_module function not found"
    exit 1
fi

if declare -f download_script >/dev/null 2>&1; then
    log_test "PASS" "download_script function available"
else
    log_test "FAIL" "download_script function not found"
    exit 1
fi

echo ""

# Test 2: Test fallback mechanism with existing local file
echo "Test 2: Testing fallback with local Redis module..."
redis_module_path="modules/cache/redis.sh"

if [[ -f "$SCRIPT_DIR/$redis_module_path" ]]; then
    log_test "PASS" "Local Redis module exists: $SCRIPT_DIR/$redis_module_path"
    
    echo "Attempting to load module with non-existent remote server..."
    if load_remote_module "$redis_module_path" 2>&1; then
        log_test "PASS" "Module loaded via fallback mechanism"
        
        # Check if Redis functions are available
        if declare -f install_redis >/dev/null 2>&1; then
            log_test "PASS" "Redis functions are available after fallback"
        else
            log_test "WARN" "Redis functions not found (might be expected)"
        fi
    else
        log_test "FAIL" "Failed to load module via fallback"
    fi
else
    log_test "FAIL" "Local Redis module not found: $SCRIPT_DIR/$redis_module_path"
fi

echo ""

# Test 3: Test with non-existent module
echo "Test 3: Testing with non-existent module..."
fake_module="modules/nonexistent/fake.sh"

echo "Attempting to load non-existent module..."
if load_remote_module "$fake_module" 2>&1; then
    log_test "FAIL" "Should have failed for non-existent module"
else
    log_test "PASS" "Correctly failed for non-existent module"
fi

echo ""

# Test 4: Test MariaDB module fallback
echo "Test 4: Testing MariaDB module fallback..."
mariadb_module_path="modules/database/mariadb.sh"

if [[ -f "$SCRIPT_DIR/$mariadb_module_path" ]]; then
    log_test "PASS" "Local MariaDB module exists"
    
    if load_remote_module "$mariadb_module_path" 2>&1; then
        log_test "PASS" "MariaDB module loaded via fallback"
        
        if declare -f install_mariadb >/dev/null 2>&1; then
            log_test "PASS" "MariaDB functions available"
        else
            log_test "WARN" "MariaDB functions not found"
        fi
    else
        log_test "FAIL" "Failed to load MariaDB module"
    fi
else
    log_test "FAIL" "Local MariaDB module not found"
fi

echo ""

# Test 5: Test PHP extension manager fallback
echo "Test 5: Testing PHP Extension Manager fallback..."
php_ext_module_path="scripts/php-extension-manager.sh"

if [[ -f "$SCRIPT_DIR/$php_ext_module_path" ]]; then
    log_test "PASS" "Local PHP extension manager exists"
    
    if load_remote_module "$php_ext_module_path" 2>&1; then
        log_test "PASS" "PHP extension manager loaded via fallback"
        
        if declare -f install_pecl_extension >/dev/null 2>&1; then
            log_test "PASS" "PHP extension functions available"
        else
            log_test "WARN" "PHP extension functions not found"
        fi
    else
        log_test "FAIL" "Failed to load PHP extension manager"
    fi
else
    log_test "FAIL" "Local PHP extension manager not found"
fi

echo ""

# Cleanup
echo "Cleaning up..."
rm -rf "$SCRIPTS_CACHE_DIR"

# Summary
echo "🎯 Test Summary"
echo "==============="
log_test "INFO" "Fallback mechanism tested"
log_test "INFO" "Script can now work offline using local files"
echo ""
echo "✨ Key Features Verified:"
echo "   • Remote download attempts first"
echo "   • Automatic fallback to local files"
echo "   • Proper error handling for missing files"
echo "   • Function availability after loading" 