#!/bin/bash

# Simple download test
echo "=== Testing Download Functions ==="

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

# Test configuration
SCRIPTS_BASE_URL="http://scripts.thanglq.dev"
SCRIPTS_CACHE_DIR="/tmp/test-thanglq-simple"
AUTO_UPDATE_SCRIPTS="true"

log_test() {
    local level=$1
    local message=$2
    case $level in
        "PASS") echo -e "${GREEN}✅ $message${NC}" ;;
        "FAIL") echo -e "${RED}❌ $message${NC}" ;;
        "INFO") echo -e "${BLUE}ℹ️  $message${NC}" ;;
        "WARN") echo -e "${YELLOW}⚠️  $message${NC}" ;;
    esac
}

# Basic download function
download_file_test() {
    local url=$1
    local output=$2
    local retries=3
    local count=0
    
    mkdir -p "$(dirname "$output")"
    
    while [[ $count -lt $retries ]]; do
        echo "Attempt $((count+1)): curl -f -s -L --connect-timeout 10 --max-time 30 '$url' -o '$output'"
        if curl -f -s -L --connect-timeout 10 --max-time 30 "$url" -o "$output"; then
            echo "Download successful!"
            return 0
        else
            ((count++))
            echo "Download failed (attempt $count/$retries)"
            sleep 2
        fi
    done
    
    return 1
}

# Test 1: Check internet connectivity
echo "Test 1: Internet connectivity"
if ping -c 1 8.8.8.8 >/dev/null 2>&1; then
    log_test "PASS" "Internet connectivity OK"
else
    log_test "FAIL" "No internet connectivity"
    exit 1
fi

# Test 2: Check curl availability
echo ""
echo "Test 2: curl availability"
if command -v curl >/dev/null 2>&1; then
    log_test "PASS" "curl is available"
    curl --version | head -1
else
    log_test "FAIL" "curl not available"
    exit 1
fi

# Test 3: Test server accessibility
echo ""
echo "Test 3: Server accessibility"
test_url="$SCRIPTS_BASE_URL/config.conf"
echo "Testing URL: $test_url"

if curl -f -s -I --connect-timeout 10 --max-time 30 "$test_url" >/dev/null 2>&1; then
    log_test "PASS" "Server is accessible"
else
    log_test "FAIL" "Server not accessible or file not found"
    echo "Response headers:"
    curl -I --connect-timeout 10 --max-time 30 "$test_url" 2>&1 | head -5
fi

# Test 4: Try downloading a simple file
echo ""
echo "Test 4: Download test"
test_file="$SCRIPTS_CACHE_DIR/config.conf"
echo "Downloading to: $test_file"

if download_file_test "$SCRIPTS_BASE_URL/config.conf" "$test_file"; then
    if [[ -f "$test_file" ]]; then
        file_size=$(ls -la "$test_file" | awk '{print $5}')
        log_test "PASS" "File downloaded successfully (size: $file_size bytes)"
        
        if [[ -s "$test_file" ]]; then
            log_test "PASS" "File is not empty"
            echo "First few lines:"
            head -5 "$test_file" 2>/dev/null || echo "Cannot read file content"
        else
            log_test "FAIL" "File is empty"
        fi
    else
        log_test "FAIL" "File was not created"
    fi
else
    log_test "FAIL" "Download failed"
fi

# Test 5: Try downloading a module
echo ""
echo "Test 5: Module download test"
module_file="$SCRIPTS_CACHE_DIR/modules/cache/redis.sh"
echo "Downloading module to: $module_file"

if download_file_test "$SCRIPTS_BASE_URL/modules/cache/redis.sh" "$module_file"; then
    if [[ -f "$module_file" ]]; then
        file_size=$(ls -la "$module_file" | awk '{print $5}')
        log_test "PASS" "Module downloaded successfully (size: $file_size bytes)"
        
        if [[ -s "$module_file" ]]; then
            log_test "PASS" "Module file is not empty"
            
            # Check if it's executable
            chmod +x "$module_file"
            if [[ -x "$module_file" ]]; then
                log_test "PASS" "Module file is executable"
            else
                log_test "WARN" "Module file is not executable"
            fi
            
            # Check syntax
            if bash -n "$module_file" 2>/dev/null; then
                log_test "PASS" "Module has valid bash syntax"
            else
                log_test "FAIL" "Module has syntax errors"
            fi
        else
            log_test "FAIL" "Module file is empty"
        fi
    else
        log_test "FAIL" "Module file was not created"
    fi
else
    log_test "FAIL" "Module download failed"
fi

# Test 6: List what we have
echo ""
echo "Test 6: Cache contents"
if [[ -d "$SCRIPTS_CACHE_DIR" ]]; then
    echo "Cache directory contents:"
    find "$SCRIPTS_CACHE_DIR" -type f -exec ls -la {} \; 2>/dev/null || echo "No files found"
else
    log_test "FAIL" "Cache directory not created"
fi

# Cleanup
echo ""
echo "Cleaning up..."
rm -rf "$SCRIPTS_CACHE_DIR"
log_test "INFO" "Test completed" 