#!/bin/bash

# ================================================================
# ThangLQ Stack - Common Utilities
# Shared functions and variables across all modules
# ================================================================

# Load configuration
load_config() {
    local config_file="${1:-config.conf}"
    if [[ -f "$config_file" ]]; then
        source "$config_file"
    else
        echo "Error: Configuration file $config_file not found"
        exit 1
    fi
}

# Color definitions
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
PURPLE='\033[0;35m'
CYAN='\033[0;36m'
WHITE='\033[1;37m'
NC='\033[0m' # No Color

# System detection
detect_system() {
    if [[ -f /etc/os-release ]]; then
        . /etc/os-release
        OS=$NAME
        VER=$VERSION_ID
        OS_FAMILY=$ID
    fi
}

# Create necessary directories
create_directories() {
    mkdir -p "$STACK_DIR" "$LOG_DIR" "$CONFIG_DIR" "$BACKUP_DIR"
    mkdir -p "$VHOSTS_DIR" "$CACHE_DIR" "$SCRIPTS_CACHE_DIR"
    mkdir -p /var/lib/php/session
}

# Download and cache script
download_script() {
    local script_path=$1
    local local_path="$SCRIPTS_CACHE_DIR/$script_path"
    local remote_url="$SCRIPTS_BASE_URL/$script_path"
    
    # Create directory structure
    mkdir -p "$(dirname "$local_path")"
    
    # Check if script exists and is recent (if auto-update enabled)
    if [[ -f "$local_path" ]] && [[ "$AUTO_UPDATE_SCRIPTS" != "true" ]]; then
        echo "$local_path"
        return 0
    fi
    
    # Check if script is older than 1 hour (if auto-update enabled)
    if [[ -f "$local_path" ]] && [[ "$AUTO_UPDATE_SCRIPTS" == "true" ]]; then
        local file_age=$(($(date +%s) - $(stat -c %Y "$local_path" 2>/dev/null || echo 0)))
        if [[ $file_age -lt 3600 ]]; then
            echo "$local_path"
            return 0
        fi
    fi
    
    log_message "INFO" "Downloading script: $script_path"
    
    # Download script with timeout and retry
    local retry_count=0
    local max_retries=3
    
    while [[ $retry_count -lt $max_retries ]]; do
        if curl -f -s -L --connect-timeout 10 --max-time 30 "$remote_url" -o "$local_path"; then
            chmod +x "$local_path"
            log_message "SUCCESS" "Downloaded: $script_path"
            echo "$local_path"
            return 0
        else
            ((retry_count++))
            log_message "WARNING" "Download failed (attempt $retry_count/$max_retries): $script_path"
            sleep 2
        fi
    done
    
    log_message "ERROR" "Failed to download script after $max_retries attempts: $script_path"
    return 1
}

# Load remote module
load_remote_module() {
    local module_path=$1
    local script_file
    
    script_file=$(download_script "$module_path")
    if [[ $? -eq 0 ]] && [[ -f "$script_file" ]]; then
        source "$script_file"
        log_message "INFO" "Loaded remote module: $module_path"
        return 0
    else
        log_message "ERROR" "Failed to load remote module: $module_path"
        return 1
    fi
}

# Check script availability
check_script_availability() {
    local script_path=$1
    local remote_url="$SCRIPTS_BASE_URL/$script_path"
    
    if curl -f -s -I --connect-timeout 5 --max-time 10 "$remote_url" >/dev/null 2>&1; then
        return 0
    else
        return 1
    fi
}

# List available remote scripts
list_available_scripts() {
    local index_url="$SCRIPTS_BASE_URL/index.txt"
    
    log_message "INFO" "Fetching available scripts list..."
    
    if curl -f -s --connect-timeout 10 --max-time 30 "$index_url" 2>/dev/null; then
        return 0
    else
        log_message "WARNING" "Could not fetch scripts index"
        return 1
    fi
}

# Clear scripts cache
clear_scripts_cache() {
    log_message "INFO" "Clearing scripts cache..."
    rm -rf "$SCRIPTS_CACHE_DIR"
    mkdir -p "$SCRIPTS_CACHE_DIR"
    log_message "SUCCESS" "Scripts cache cleared"
}

# Logging function
log_message() {
    local level=$1
    local message=$2
    local timestamp=$(date '+%Y-%m-%d %H:%M:%S')
    local log_file="${LOG_DIR}/install.log"
    
    # Ensure log directory exists
    mkdir -p "$LOG_DIR"
    
    echo "[$timestamp] [$level] $message" >> "$log_file"
    
    case $level in
        "ERROR")
            echo -e "${RED}[ERROR]${NC} $message"
            ;;
        "SUCCESS")
            echo -e "${GREEN}[SUCCESS]${NC} $message"
            ;;
        "WARNING")
            echo -e "${YELLOW}[WARNING]${NC} $message"
            ;;
        "INFO")
            echo -e "${BLUE}[INFO]${NC} $message"
            ;;
        "DEBUG")
            if [[ "$LOG_LEVEL" == "DEBUG" ]]; then
                echo -e "${PURPLE}[DEBUG]${NC} $message"
            fi
            ;;
    esac
}

# Check if running as root
check_root() {
    if [[ $EUID -ne 0 ]]; then
        log_message "ERROR" "Script must be run as root"
        exit 1
    fi
}

# Check internet connectivity
check_internet() {
    log_message "INFO" "Checking internet connectivity..."
    
    if ping -c 1 8.8.8.8 >/dev/null 2>&1 || ping -c 1 1.1.1.1 >/dev/null 2>&1; then
        log_message "SUCCESS" "Internet connectivity confirmed"
        return 0
    else
        log_message "ERROR" "No internet connectivity"
        return 1
    fi
}

# System requirements check
check_requirements() {
    log_message "INFO" "Checking system requirements..."
    
    # Check OS compatibility
    detect_system
    if [[ ! "$OS_FAMILY" =~ ^(almalinux|rocky|centos|rhel)$ ]]; then
        log_message "ERROR" "Unsupported OS. This script supports AlmaLinux, Rocky Linux, CentOS, RHEL"
        exit 1
    fi
    
    # Check memory
    local memory_kb=$(grep MemTotal /proc/meminfo | awk '{print $2}')
    local memory_gb=$((memory_kb / 1024 / 1024))
    
    if [[ $memory_gb -lt 2 ]]; then
        log_message "WARNING" "System has less than 2GB RAM. Minimum 4GB recommended"
    fi
    
    # Check disk space
    local disk_space=$(df / | awk 'NR==2 {print $4}')
    local disk_gb=$((disk_space / 1024 / 1024))
    
    if [[ $disk_gb -lt 10 ]]; then
        log_message "ERROR" "Insufficient disk space. At least 10GB required"
        exit 1
    fi
    
    # Check internet connectivity
    if ! check_internet; then
        log_message "ERROR" "Internet connection required for downloading scripts"
        exit 1
    fi
    
    # Check curl availability
    if ! command -v curl >/dev/null 2>&1; then
        log_message "INFO" "Installing curl..."
        if command -v dnf >/dev/null 2>&1; then
            dnf install -y curl
        else
            yum install -y curl
        fi
    fi
    
    log_message "SUCCESS" "System requirements check passed"
}

# Install development tools
install_development_tools() {
    log_message "INFO" "Installing development tools..."
    
    if command -v dnf >/dev/null 2>&1; then
        dnf groupinstall -y "Development Tools"
        dnf install -y epel-release wget curl git gcc gcc-c++ make cmake \
            pcre-devel zlib-devel openssl-devel libxml2-devel \
            libxslt-devel gd-devel perl-ExtUtils-Embed GeoIP-devel \
            libatomic_ops-devel gperftools-devel libunwind-devel \
            jemalloc-devel libevent-devel systemd-devel
    else
        yum groupinstall -y "Development Tools"
        yum install -y epel-release wget curl git gcc gcc-c++ make cmake \
            pcre-devel zlib-devel openssl-devel libxml2-devel \
            libxslt-devel gd-devel perl-ExtUtils-Embed GeoIP-devel \
            libatomic_ops-devel gperftools-devel libunwind-devel \
            jemalloc-devel libevent-devel systemd-devel
    fi
    
    log_message "SUCCESS" "Development tools installed"
}

# Get system information
get_system_info() {
    # Memory information
    TOTAL_RAM=$(free -m | awk 'NR==2{printf "%.0f", $2}')
    AVAILABLE_RAM=$(free -m | awk 'NR==2{printf "%.0f", $7}')
    
    # CPU information
    CPU_CORES=$(nproc)
    CPU_THREADS=$(grep -c ^processor /proc/cpuinfo)
    
    # Disk information
    DISK_TYPE="SSD"
    if [[ -f /sys/block/sda/queue/rotational ]] && [[ $(cat /sys/block/sda/queue/rotational) == "1" ]]; then
        DISK_TYPE="HDD"
    fi
    
    log_message "INFO" "System Info: ${TOTAL_RAM}MB RAM, ${CPU_CORES} cores, ${DISK_TYPE}"
}

# Check if service is running
check_service() {
    local service_name=$1
    if systemctl is-active --quiet "$service_name"; then
        return 0
    else
        return 1
    fi
}

# Backup file with timestamp
backup_file() {
    local file_path=$1
    local backup_dir="${BACKUP_DIR}/configs"
    
    if [[ -f "$file_path" ]]; then
        mkdir -p "$backup_dir"
        local filename=$(basename "$file_path")
        local backup_name="${filename}.backup.$(date +%Y%m%d_%H%M%S)"
        cp "$file_path" "${backup_dir}/${backup_name}"
        log_message "INFO" "Backed up $file_path to ${backup_dir}/${backup_name}"
    fi
}

# Test configuration file
test_config() {
    local service=$1
    case $service in
        "nginx")
            nginx -t
            ;;
        "php-fpm")
            php-fpm -t
            ;;
        *)
            log_message "WARNING" "Unknown service for config test: $service"
            return 1
            ;;
    esac
}

# Restart service safely
restart_service() {
    local service_name=$1
    log_message "INFO" "Restarting $service_name..."
    
    if systemctl restart "$service_name"; then
        log_message "SUCCESS" "$service_name restarted successfully"
        return 0
    else
        log_message "ERROR" "Failed to restart $service_name"
        return 1
    fi
}

# Show banner
show_banner() {
    echo -e "${CYAN}"
    echo "========================================================"
    echo "            $BRAND_NAME"
    echo "                   Version $BRAND_VERSION"
    echo "                   $BRAND_URL"
    echo "========================================================"
    echo -e "${NC}"
}

# Progress indicator
show_progress() {
    local current=$1
    local total=$2
    local message=$3
    local percent=$((current * 100 / total))
    
    echo -ne "\r${BLUE}[INFO]${NC} Progress: [$(printf "%*s" $((percent/2)) | tr ' ' '=')]$(printf "%*s" $((50-percent/2)) | tr ' ' '-') ${percent}% - $message"
    
    if [[ $current -eq $total ]]; then
        echo ""
    fi
}

# Validate PHP version format
validate_php_version() {
    local version=$1
    if [[ $version =~ ^[78]\.[0-9]+$ ]]; then
        return 0
    else
        log_message "ERROR" "Invalid PHP version format: $version"
        return 1
    fi
}

# Check if PHP version is installed
check_php_version() {
    local php_version=$1
    local php_binary="/usr/local/php${php_version}/bin/php"
    
    if [[ -f "$php_binary" ]]; then
        return 0
    else
        return 1
    fi
}

# Get available PHP versions
get_available_php_versions() {
    echo "$PHP_AVAILABLE_VERSIONS"
}

# Create user if not exists
create_user() {
    local username=$1
    local home_dir=$2
    
    if ! id "$username" >/dev/null 2>&1; then
        if [[ -n "$home_dir" ]]; then
            useradd -r -d "$home_dir" -s /bin/false "$username"
        else
            useradd -r -s /bin/false "$username"
        fi
        log_message "SUCCESS" "Created user: $username"
    else
        log_message "INFO" "User $username already exists"
    fi
}

# Generate random password
generate_password() {
    local length=${1:-16}
    openssl rand -base64 $length | tr -d "=+/" | cut -c1-$length
}

# Install from source
install_from_source() {
    local name=$1
    local url=$2
    local configure_args=$3
    local version=$4
    
    log_message "INFO" "Installing $name $version from source..."
    
    cd /tmp
    
    # Download
    local filename="${name}-${version}.tar.gz"
    if [[ ! -f "$filename" ]]; then
        wget "$url" -O "$filename"
        if [[ $? -ne 0 ]]; then
            log_message "ERROR" "Failed to download $name"
            return 1
        fi
    fi
    
    # Extract
    tar -xzf "$filename"
    cd "${name}-${version}" || return 1
    
    # Configure
    if [[ -n "$configure_args" ]]; then
        ./configure $configure_args
    else
        ./configure
    fi
    
    if [[ $? -ne 0 ]]; then
        log_message "ERROR" "$name configure failed"
        return 1
    fi
    
    # Compile
    make -j$(nproc)
    if [[ $? -ne 0 ]]; then
        log_message "ERROR" "$name compilation failed"
        return 1
    fi
    
    # Install
    make install
    if [[ $? -ne 0 ]]; then
        log_message "ERROR" "$name installation failed"
        return 1
    fi
    
    log_message "SUCCESS" "$name $version installed successfully"
    return 0
}

# Check if module is enabled
is_module_enabled() {
    local module=$1
    local var_name="${module^^}_MODULE_ENABLED"
    local enabled="${!var_name}"
    
    if [[ "$enabled" == "true" ]]; then
        return 0
    else
        return 1
    fi
}

# Load module if enabled
load_module() {
    local module_name=$1
    local module_path="modules/$module_name"
    
    if is_module_enabled "$module_name" && [[ -f "$module_path/${module_name}.sh" ]]; then
        source "$module_path/${module_name}.sh"
        log_message "INFO" "Loaded module: $module_name"
        return 0
    else
        log_message "DEBUG" "Module $module_name not enabled or not found"
        return 1
    fi
} 